--[[
Ideas/Notes:
- think about whether the kickback can actually be described as a "Dodge" -- and proc "Dodge" powers??

short-term TODO:
- work on visualizing focus-clip mechanic
]]

local SGCommon = require "stategraphs.sg_common"
local SGPlayerCommon = require "stategraphs.sg_player_common"
local fmodtable = require "defs.sound.fmodtable"
local combatutil = require "util.combatutil"
local soundutil = require "util.soundutil"
local kassert = require "util.kassert"
local krandom = require "util.krandom"
local EffectEvents = require "effectevents"
local ParticleSystemHelper = require "util.particlesystemhelper"
local Weight = require "components/weight"
local powerutil = require "util.powerutil"
local Equipment = require "defs.equipment"

local ATTACKS =
{
	SINGLE =
	{
		DAMAGE = 0.75,
		DAMAGE_FOCUS = 1.5,
		HITSTUN = 3,
		HITSTUN_FOCUS = 5,
		PUSHBACK = 0,
		PUSHBACK_FOCUS = 0,
		SPEED = 18,
		SPEED_FOCUS = 20,
		RANGE = 20,
		RANGE_FOCUS = 20,
	},

	BLAST =
	{
		DAMAGE = 0.25, --PROPOSED TUNING: 0.25
		DAMAGE_FOCUS = 0.75, --PROPOSED TUNING: 0.5 or 0.75
		HITSTUN = 10,
		HITSTUN_FOCUS = 15,
		PUSHBACK = 0.75,
		PUSHBACK_FOCUS = 1,
		SPEED = 22,
		SPEED_FOCUS = 24,
		RANGE = 10,
		RANGE_FOCUS = 10,
	},

	QUICKRISE =
	{
		DAMAGE = 0.75,
		DAMAGE_FOCUS = 1.5,
		HITSTUN = 1,
		HITSTUN_FOCUS = 15,
		PUSHBACK = 1,
		PUSHBACK_FOCUS = 2,
		HITSTOP = HitStopLevel.MEDIUM, -- Some hitstop is already applied by the move itself, this is additional hitstop
		RADIUS = 3,
	},

	MORTAR =
	{
		DAMAGE = 2, -- 12 total for full 6-ammo shot
		DAMAGE_FOCUS = 3, -- 9 total for full 3-ammo shot
		HITSTUN = 10,
		HITSTUN_FOCUS = 15,
		PUSHBACK = 1,
		PUSHBACK_FOCUS = 2,
		HITSTOP = HitStopLevel.HEAVY,
		RADIUS = 2,
		RADIUS_FOCUS = 3,
	},

	SHOCKWAVE_WEAK =
	{
		DAMAGE = 2,
		DAMAGE_FOCUS = 3,
		HITSTUN = 8,
		HITSTUN_FOCUS = 10,
		PUSHBACK = 1,
		PUSHBACK_FOCUS = 1.5,
		HITSTOP = HitStopLevel.MEDIUM,
		RADIUS = 3,
		KNOCKDOWN = false,
		SELF_DAMAGE = 0,
		SELF_HIT_FX_X_OFFSET = 0.5,
		SELF_HIT_FX_Y_OFFSET = 2.5,
	},

	SHOCKWAVE_MEDIUM =
	{
		DAMAGE = 3,
		DAMAGE_FOCUS = 4,
		HITSTUN = 10,
		HITSTUN_FOCUS = 12,
		PUSHBACK = 1.5,
		PUSHBACK_FOCUS = 2,
		HITSTOP = HitStopLevel.MEDIUM,
		RADIUS = 3.5,
		KNOCKDOWN = true,
		KNOCKDOWN_RADIUS = 0.8, -- What percentage of "Radius" above should create a KNOCKDOWN, instead of just a knockback?
		SELF_DAMAGE = 0,
		SELF_HIT_FX_X_OFFSET = 0.5,
		SELF_HIT_FX_Y_OFFSET = 2.5,
	},

	SHOCKWAVE_STRONG =
	{
		-- This attack cannot be FOCUS. Only shots with remaining ammo 3,2,1 are focus. Leaving tuning in case something weird happens.
		DAMAGE = 5,
		SELF_DAMAGE = 30,
		DAMAGE_FOCUS = 6,
		HITSTUN = 15,
		HITSTUN_FOCUS = 15,
		PUSHBACK = 2,
		PUSHBACK_FOCUS = 3,
		HITSTOP = HitStopLevel.HEAVY,
		RADIUS = 5,
		KNOCKDOWN = true,
		KNOCKDOWN_RADIUS = 1.0, -- What percentage of "Radius" above should create a KNOCKDOWN, instead of just a knockback?
		SELF_HIT_FX_X_OFFSET = 0.5,
		SELF_HIT_FX_Y_OFFSET = 2.5,
	},

	BACKFIRE_WEAK =
	{
		DAMAGE = 0.5,
		DAMAGE_FOCUS = .75,
		HITSTUN = 4,
		HITSTUN_FOCUS = 4,
		PUSHBACK = 0.5,
		PUSHBACK_FOCUS = 1,
		HITSTOP = HitStopLevel.NONE,

		KNOCK = "KNOCKBACK",

		SELF_DAMAGE = 0,
		SELF_HIT_FX_X_OFFSET = -2,
		SELF_HIT_FX_Y_OFFSET = 1,
	},

	BACKFIRE_MEDIUM_EARLY =
	{
		-- This attack is for the first, initial blast of the backfire attack. The slower part towards the end is BACKFIRE_MEDIUM_LATE
		DAMAGE = 3,
		DAMAGE_FOCUS = 4,
		HITSTUN = 10,
		HITSTUN_FOCUS = 10,
		PUSHBACK = 2,
		PUSHBACK_FOCUS = 2,
		HITSTOP = HitStopLevel.NONE,

		KNOCK = "KNOCKDOWN",

		SELF_DAMAGE = 0,
		SELF_HIT_FX_X_OFFSET = -2,
		SELF_HIT_FX_Y_OFFSET = 1,
	},

	BACKFIRE_MEDIUM_LATE =
	{
		-- Just the slowdown of BACKFIRE_MEDIUM
		DAMAGE = 4,
		DAMAGE_FOCUS = 5,
		HITSTUN = 1,
		HITSTUN_FOCUS = 1,
		PUSHBACK = 0.5,
		PUSHBACK_FOCUS = 1,
		HITSTOP = HitStopLevel.NONE,

		KNOCK = "KNOCKBACK",

		SELF_DAMAGE = 0,
		SELF_HIT_FX_X_OFFSET = -2,
		SELF_HIT_FX_Y_OFFSET = 1,
	},

	BACKFIRE_STRONG_EARLY =
	{
		-- This attack is for the first, initial blast of the backfire attack. The body landing on the ground is "BACKFIRE_STRONG_LATE"
		-- This attack cannot be FOCUS. Only shots with remaining ammo 3,2,1 are focus. Leaving tuning in case something weird happens.
		DAMAGE = 4,
		DAMAGE_FOCUS = 5,
		HITSTUN = 10,
		HITSTUN_FOCUS = 10,
		PUSHBACK = 2,
		PUSHBACK_FOCUS = 2,
		HITSTOP = HitStopLevel.HEAVY,

		KNOCK = "KNOCKDOWN",

		SELF_DAMAGE = 0,
		SELF_HIT_FX_X_OFFSET = -2,
		SELF_HIT_FX_Y_OFFSET = 1,
	},

	BACKFIRE_STRONG_LATE =
	{
		-- This attack is for the player's body landing on the ground. Don't push back very much, and don't do much hitstun. If they are landing in danger, they should not have frame advantage.
		-- This attack cannot be FOCUS. Only shots with remaining ammo 3,2,1 are focus. Leaving tuning in case something weird happens.
		DAMAGE = 1,
		DAMAGE_FOCUS = 2,
		HITSTUN = 1,
		HITSTUN_FOCUS = 1,
		PUSHBACK = 0.5,
		PUSHBACK_FOCUS = 0.5,
		HITSTOP = HitStopLevel.LIGHT,

		KNOCK = "KNOCKBACK",

		SELF_DAMAGE = 30,
		SELF_HIT_FX_X_OFFSET = -2,
		SELF_HIT_FX_Y_OFFSET = 1,
	},
}

local WEIGHT_TO_DODGE_MULTIPLIER =
{
	[Weight.Status.s.Normal] = 1,
	[Weight.Status.s.Light] = 1.5,
	[Weight.Status.s.Heavy] = 0.3, -- Keeps Heavy player up close, very offensive. They can do multiple shotgun blasts against the same enemy.
}
local WEIGHT_TO_BACKFIRE_MULTIPLIER =
{
	-- This is such a strong movement that we should be more specific about it. Using the values above affect the move so severely.
	[Weight.Status.s.Normal] = 1,
	[Weight.Status.s.Light] = 1.25,
	[Weight.Status.s.Heavy] = 0.75,
}

-- MORTAR PARAMETERS:
local MORTAR_AIM_START_DISTANCE = 7 -- How far away from the player does the mortar aim indicator start?
local MORTAR_AIM_MAX_DISTANCE = 15 -- How far away from the player is the max distance a mortar can be aimed to?
local MORTAR_SHOOT_HITSTOP =
{
	WEAK = HitStopLevel.LIGHT,
	MEDIUM = HitStopLevel.MEDIUM,
	STRONG = HitStopLevel.MEDIUM,
}
local MORTAR_SHOOT_BLOWBACK = -- How much is the player blown back when doing a weak, medium, or strong mortar blast?
{
	WEAK = 0,
	MEDIUM = 4,
	STRONG = 10,
}

local MORTAR_AIM_RETICLE_SPEED = 0.1
local MORTAR_AIM_OFFSET =
{
	-- When shooting multiple projectiles in a mortar, what positioning should they have?
	{ x =  0,  z = 0  },
	{ x =  -2,  z = 0.5  },
	{ x =  2,  z = 0.5  },
	{ x =  -1,  z = -2  },
	{ x =  1,  z = -2  },
	{ x =  0,  z = 2  },
}

local MORTAR_SHOOT_TIMING =
{
	-- When firing a mortar, how many frames should a given bullet be delayed, so they don't shoot all at once??
	0,
	0,
	2,
	2,
	4,
	4,
}

-- We randomly scale each mortar's anim to create offsets/visual variation. What's the min/max?
local MORTAR_RANDOM_SCALE_MIN = 0.7
local MORTAR_RANDOM_SCALE_MAX = 1.0

-- We randomly scale the speed of each mortar's anim to rotate at different speeds. The animation speed is the rotation speed. What's the min/max?
local MORTAR_RANDOM_ROTATESPEED_MIN = 0.5
local MORTAR_RANDOM_ROTATESPEED_MAX = 1.5

-- Backfire Parameters:
local BACKFIRE_VELOCITY =
{
	WEAK = -28,
	MEDIUM = -34,
	STRONG = -34,
}

local BLAST_RECOIL_FRAME_TO_SPEEDMULT =
{
	-- Doing it this way so that I can count how many frames we've been sliding across multiple possible states.
	-- These are SetMotorVel()s done in update of any state that is said to be "dodging"
	-- For example, if I BLAST backwards, and cancel into a SHOT or a PLANT -- maintain this tightly tuned velocity the whole way through.
	-- cannon_H_atk
	[1] = 1.75,
	[4] = 1,
	[8] = 1,
	-- cannon_H_land
	[9] = 0.5,
	[10] = 0.45,
	[11] = 0.4 ,
	[12] = 0.35,
	[13] = 0.3,
	[14] = 0.15,
	[15] = 0,
	[16] = 0,
}

-- number of bullets remaining below which we trigger a low ammo sound
-- note that this is generally always checked AFTER firing
local LOW_AMMO_THRESHOLDS =
{
	[6] = 3,
	[5] = 2,
	[4] = 2,
	[3] = 1,
	[2] = 1,
	[1] = 1,
}

-- AMMO MANAGEMENT
local function UpdateAmmoSymbols(inst)

	local ammo = inst.sg.mem.ammo
	local focus = inst.sg.mem.focus_sequence and inst.sg.mem.focus_sequence[ammo] or false
	-- Glow the tip of the weapon blue when focus section is active.
	if focus then
		inst.AnimState:SetSymbolBloom("feature01", 0/255, 100/255, 150/255, 200/255)
	else
		inst.AnimState:SetSymbolBloom("feature01", 0/255, 0/255, 0/255, 200/255)
	end

	if ammo == 0 then
		local param =
		{
			name = "noammo_particles",
			particlefxname = "smoke_cannon_empty",
			followsymbol = "swap_fx",
			ischild = true,
			offz = 0.1,
		}

		ParticleSystemHelper.MakeEventSpawnParticles(inst, param)
	else
		local param =
		{
			name = "noammo_particles",
		}
		return ParticleSystemHelper.MakeEventStopParticles(inst, param)
	end
end

local function GetMaxAmmo(inst)
	return inst.sg.mem.ammo_max
end

local function GetMissingAmmo(inst)
	return inst.sg.mem.ammo_max - inst.sg.mem.ammo
end

local function GetRemainingAmmo(inst)
	return inst.sg.mem.ammo
end

local function UpdateAmmo(inst, amount)
	--sound
	local prev_ammo = inst.sg.mem.ammo
	
	inst.sg.mem.ammo = math.max(inst.sg.mem.ammo - amount, 0)

	--sound
	local change_direction = inst.sg.mem.ammo < prev_ammo and -1 or 1
	local parameter = ((prev_ammo - inst.sg.mem.ammo) / GetMaxAmmo(inst)) * change_direction
	inst.sg.statemem.percent_ammo_changed_param = parameter

	UpdateAmmoSymbols(inst)

	local lifetime_reloads = inst.components.progresstracker:GetValue("total_cannon_reloads") or 0
	if inst.sg.mem.ammo <= 0 and lifetime_reloads <= 2 then
		TheDungeon.HUD:TutorialPopup(STRINGS.UI.TUTORIAL_POPUPS.CANNON_RELOAD, inst)
	end
end

local function OnReload(inst, amount)
	local parameter
	if inst.sg.mem.ammo == inst.sg.mem.ammo_max then
		parameter = 0
	else
		parameter = math.abs(amount / GetMaxAmmo(inst))
	end

	inst.sg.statemem.percent_ammo_changed_param = parameter

	inst.sg.mem.ammo = math.min(inst.sg.mem.ammo + amount, inst.sg.mem.ammo_max)

	UpdateAmmoSymbols(inst)

	-- if a player tries to fire with an empty clip more than once (we allow for one accidental dry fire)
	-- we assume that they might need help counting ammo
	-- so the next full clip they fire will have slightly louder 'ammo counter' sounds
	-- this state is cleared when they reload, in any way, so long as they haven't hit the 'no ammo' fire state agin
	-- which might mean that they've better learned how to count their ammo
	if inst.sound_triedToFireWithoutAmmo and inst.sound_triedToFireWithoutAmmo > 1 then
		inst.makeAmmoCountSoundsLouderToAssist = true
	else
		inst.makeAmmoCountSoundsLouderToAssist = nil
	end
	inst.sound_triedToFireWithoutAmmo = nil
end

local function GetWeightVelocityMult(inst)
	local weight = inst.components.weight:GetStatus()
	local weightmult = WEIGHT_TO_DODGE_MULTIPLIER[weight]

	return weightmult
end

local function GetBackfireWeightVelocityMult(inst)
	local weight = inst.components.weight:GetStatus()
	local weightmult = WEIGHT_TO_BACKFIRE_MULTIPLIER[weight]

	return weightmult
	end
--

local function CreateMortarAimReticles(inst)
	if inst.sg.mem.aim_reticles ~= nil then
		for reticle,aim_data in pairs(inst.sg.mem.aim_reticles) do
			if reticle ~= nil and reticle:IsValid() then
				reticle:Remove()
			end
		end
	end
	inst.sg.mem.aim_reticles = {}

	-- Get some global information about inst's position and direction
	local x, z = inst.Transform:GetWorldXZ()
	local facingright = inst.Transform:GetFacing() == FACING_RIGHT
	local start_offset = facingright and MORTAR_AIM_START_DISTANCE or -MORTAR_AIM_START_DISTANCE

	-- Set up the root aim distance, so we're not relying on FX position and existence for calculating the distance
	inst.sg.mem.aim_root_x = x + start_offset
	inst.sg.mem.aim_root_z = z
	inst.sg.mem.aim_root_speed = facingright and MORTAR_AIM_RETICLE_SPEED or -MORTAR_AIM_RETICLE_SPEED

	-- For every bullet we are about to shoot, create an aim indicator
	local bullets = inst.sg.mem.cannon_override_mortar_ammopershot or GetRemainingAmmo(inst)
	for i=1,bullets do
		local circle = SpawnPrefab("fx_ground_target_player", inst)
		circle.Transform:SetScale(0.75, 0.75, 0.75)

		local aim_offset = MORTAR_AIM_OFFSET[i]
		local offsetmult = facingright and -1 or 1

		local aim_data =
		{
			aim_x = x + start_offset + aim_offset.x * offsetmult,
			aim_z = z + aim_offset.z,
			aim_speed = facingright and MORTAR_AIM_RETICLE_SPEED or -MORTAR_AIM_RETICLE_SPEED,
		}
		inst.sg.mem.aim_reticles[circle] = aim_data

		circle.Transform:SetPosition(aim_data.aim_x, 0, aim_data.aim_z)
	end

	inst.sg.mem.update_aim = true

end

local function UpdateMortarAimReticles(inst)
	local dist = inst:GetDistanceSqToXZ(inst.sg.mem.aim_root_x, inst.sg.mem.aim_root_z)

	if dist <= (MORTAR_AIM_MAX_DISTANCE * MORTAR_AIM_MAX_DISTANCE) then
		-- Only allow aiming up until the max distance

		for circle,aim_data in pairs(inst.sg.mem.aim_reticles) do
			aim_data.aim_x = aim_data.aim_x + aim_data.aim_speed
			circle.Transform:SetPosition(aim_data.aim_x, 0, aim_data.aim_z)
		end

		inst.sg.mem.aim_root_x = inst.sg.mem.aim_root_x + inst.sg.mem.aim_root_speed
	end
end

local function DestroyMortarAimReticles(inst)
	if inst ~= nil and inst.sg.mem.aim_reticles ~= nil then
		for reticle,aim_data in pairs(inst.sg.mem.aim_reticles) do
			if reticle ~= nil and reticle:IsValid() then
				reticle:Remove()
			end
		end
	end
end

local function ConfigureNewDodge(inst)
	--[[
		In order to actually start the dodge, make sure that:
			- state has DoDodgeMovement() in onupdate()
			- call StartNewDodge() on the frame you want the movement to begin
			- probably add a Kickback() function, like DoKickback or DoBlastKickback
	]]
	local weightmult = GetWeightVelocityMult(inst)
	local locomotorspeedmult = inst.components.locomotor.total_speed_mult * 0.75 --TODO: use the common dodge func

	inst.sg.statemem.maxspeed = -TUNING.GEAR.WEAPONS.CANNON.ROLL_VELOCITY * locomotorspeedmult * weightmult
	inst.sg.statemem.framessliding = 0
end
local function StartNewDodge(inst)
	inst.sg.statemem.speed = inst.sg.statemem.maxspeed
end
local function CheckIfDodging(inst, data)
	if data ~= nil then
		inst.sg.statemem.maxspeed = data.maxspeed
		inst.sg.statemem.speed = data.speed
		inst.sg.statemem.framessliding = data.framessliding

		if inst.sg.statemem.framessliding ~= nil and inst.sg.statemem.framessliding <= TUNING.PLAYER.ROLL.NORMAL.IFRAMES then -- TODO #weight make work for different weights
			inst.HitBox:SetInvincible(true)
		end
	else
		-- Assert and print the state we came from that failed to pass transitiondata:
		--~ dbassert(data, inst.sg.laststate and inst.sg.laststate.name or "<no laststate>")
		-- Trying to fix a crash I can't repro... we didn't get data here, so just set maxspeed to 0.
		-- Elsewhere, if we received maxspeed = 0 then print some logging to help identify why
		inst.sg.statemem.maxspeed = 0
		inst.sg.statemem.speed = 0
		inst.sg.statemem.framessliding = 0
		inst.Physics:Stop()
	end
end
local function DoDodgeMovement(inst)
	if not inst.sg.statemem.pausedodgemovement then
		if inst.sg.statemem.speed ~= nil and inst.sg.statemem.framessliding ~= nil then
			-- print("--------")
			-- print(inst.sg:GetCurrentState()..": "..inst.sg:GetTicksInState())
			-- print("inst.sg.statemem.speed", inst.sg.statemem.speed, "inst.sg.statemem.framessliding", inst.sg.statemem.framessliding)
			inst.sg.statemem.framessliding = inst.sg.statemem.framessliding + 0.5 -- a tick is 0.5 a 'frame'
			if BLAST_RECOIL_FRAME_TO_SPEEDMULT[inst.sg.statemem.framessliding] ~= nil then
				inst.sg.statemem.speed = inst.sg.statemem.maxspeed * BLAST_RECOIL_FRAME_TO_SPEEDMULT[inst.sg.statemem.framessliding]
			end

			inst.Physics:SetMotorVel(inst.sg.statemem.speed)
			if inst.sg.statemem.speed == 0 then
				inst.Physics:Stop()
				inst.sg.statemem.speed = nil
			end
		end

		-- BUG... this keeps invincible for way longer because air-to-air cancel states set framessliding back down to 0.
		-- Either do that movement boost in a different way OR track iframes individually
		if inst.sg.statemem.framessliding ~= nil and inst.sg.statemem.framessliding > TUNING.PLAYER.ROLL.NORMAL.IFRAMES then -- TODO #weight make work for different weights
			inst.HitBox:SetInvincible(false)
		else
			-- print("Invincible this frame:", inst.sg.statemem.framessliding)
		end
	end
end

local function DoAirShootMovement(inst)
	if not inst.sg.statemem.pausedodgemovement then
		if inst.sg.statemem.maxspeed == 0 then
			-- HACK: hopefully temporary bandaid to fix a crash when 'transitiondata' became nil at some point.
			local laststatename = inst.sg.laststate ~= nil and inst.sg.laststate.name
			print("WARNING: [Cannon] DoAirShootMovement has received inst.sg.statemem.maxspeed of 0. Configuring new dodge.", inst.sg:GetTicksInState(), laststatename ~= nil and laststatename)
			ConfigureNewDodge(inst)
		end
		inst.sg.statemem.framessliding = 8
		inst.sg.statemem.speed = inst.sg.statemem.maxspeed * BLAST_RECOIL_FRAME_TO_SPEEDMULT[inst.sg.statemem.framessliding]
	end
end

-- KICKBACK FUNCTIONS:
-- Moving the player back, sharply, when an attack is executed.
local function DoShootKickback(inst)
	inst.Physics:MoveRelFacing(-25 / 150)
end

local function DoBlastKickback(inst)
	inst.Physics:MoveRelFacing(-125 / 150)
end

local function DoAirBlastKickback(inst)
	inst.Physics:MoveRelFacing(-250 / 150)
end

local function DoQuickRiseKickback(inst)
	inst.Physics:MoveRelFacing(-150 / 150)
end

local function DoMortarWeakKickback(inst)
	inst.Physics:MoveRelFacing(-25 / 150)
end

local function DoMortarMediumKickback(inst)
	inst.Physics:MoveRelFacing(-50 / 150)
end

local function DoMortarStrongKickback(inst)
	inst.Physics:MoveRelFacing(-150 / 150)
end

local function DoBackfireWeakKickback(inst)
	inst.Physics:MoveRelFacing(-75 / 150)
end

local function DoBackfireMediumKickback(inst)
	inst.Physics:MoveRelFacing(-150 / 150)
end

local function DoBackfireStrongKickback(inst)
	inst.Physics:MoveRelFacing(-250 / 150)
end

-- SOUND
local function IsShotOnLowAmmo(inst, current_ammo)
	local current_ammo = current_ammo or GetRemainingAmmo(inst)
	-- must be called AFTER AMMO IS UPDATED
	local threshold
	
	if LOW_AMMO_THRESHOLDS[GetMaxAmmo(inst)] then
		threshold = LOW_AMMO_THRESHOLDS[GetMaxAmmo(inst)]
	else
		--some scaling for future weapons
		local base_scale = 1/3 -- for low ish max ammo counts
		local adjusted_scale = 1/10 -- for much higher future max ammo counts like, 100

		local startAdjustingAt = 30 -- start adjusting after GetMaxAmmo(inst) is greater this value
		local adjustOverRange = 70

		if GetMaxAmmo <= startAdjustingAt then
			return math.floor((GetMaxAmmo * base_scale) + 0.5)
		elseif GetMaxAmmo >= (startAdjustingAt + adjustOverRange) then
			-- for values much higher than startAdjustingAt, use the adjusted scale
			return math.floor((GetMaxAmmo * adjusted_scale) + 0.5)
		else
			-- for startAdjustingAt < GetMaxAmmo(inst) < adjustOverRange
			local progress = (GetMaxAmmo - startAdjustingAt) / adjustOverRange
			local scale = base_scale * (1 - progress) + adjusted_scale * progress
			return math.floor((GetMaxAmmo * scale) + 0.5)
		end

		threshold = math.floor((GetMaxAmmo(inst)/3) + 0.5) -- 
	end
	
	local is_low_ammo = current_ammo < threshold
	local parameter = current_ammo / threshold
	-- print("IsShotOnLowAmmo", current_ammo, threshold, is_low_ammo, parameter)

	return is_low_ammo, parameter
end

local function PlayLowAmmoSound(inst, current_ammo, volume)
	volume = volume or 100
	assert(type(volume) == "number", "volume must be a number")
	if volume <= 1 then
		volume = volume * 100
	end

	-- must be called AFTER AMMO IS UPDATED
	local is_low_ammo, parameter = IsShotOnLowAmmo(inst, current_ammo)
	if not is_low_ammo then
		return
	end

	if inst.sg.mem.ammo_sound then
		soundutil.KillSound(inst, inst.sg.mem.ammo_sound)
		inst.sg.mem.ammo_sound = nil
	end

	inst.sg.mem.ammo_sound = soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_ammoCounter, {
		volume = volume,
		max_count = 1,
		fmodparams = {
			cannon_ammo_numLowAmmoShotsRemaining_scaled = parameter,
			cannon_ammo_makeSoundsLouderToAssist = inst.makeAmmoCountSoundsLouderToAssist and 1 or 0,
		}
	})
end

local function PlayMortarSound(inst, ammo_cost, cannonMortarStrength)
	-- have to do this because we are playing this sound before the ammo updates
	local cannon_ammo_percentDelta = (ammo_cost / GetMaxAmmo(inst)) * -1
	local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.focus)
	soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_mortar_launch_fire_scatterer, {
		max_count = 1,
		fmodparams = {
			isFocusAttack = isFocusAttack,
			numBombs = ammo_cost,
			cannon_mortarStrength = cannonMortarStrength,
			cannon_ammo_percentDelta = cannon_ammo_percentDelta,
		}
	})
end

local function PlayMortarTubeSound(inst, cannonMortarStrength, isFocusAttack)
	soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_mortar_launch_tube, {
		max_count = 1,
		fmodparams = {
			isFocusAttack = isFocusAttack,
			cannon_mortarStrength = cannonMortarStrength,
		}
	})
end

local function PlayNoAmmoSound(inst)
	soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_noammo,{ max_count = 1 })
	inst.sound_triedToFireWithoutAmmo = (inst.sound_triedToFireWithoutAmmo or 0) + 1
	local volume = inst.sound_triedToFireWithoutAmmo > 1 and 1 or .7
	inst:DoTaskInAnimFrames(4,function() PlayLowAmmoSound(inst,0,volume) end)
end
--

-- FUNCTIONS FOR DOING SHOTS
local function DoShoot(inst)
	-- First, initialize the attack
	local ATTACK =  ATTACKS.SINGLE

	local damagemod
	local hitstun
	local pushback
	local speed
	local range
	local projectileprefab

	local remaining_ammo = GetRemainingAmmo(inst)

	-- If this is a focus attack, use FOCUS numbers. Otherwise, use default numbers.
	local focus = inst.sg.mem.focus_sequence[remaining_ammo]
	if focus then
		damagemod = ATTACK.DAMAGE_FOCUS
		hitstun = ATTACK.HITSTUN_FOCUS
		pushback = ATTACK.PUSHBACK_FOCUS
		speed = ATTACK.SPEED_FOCUS
		range = ATTACK.RANGE_FOCUS
		projectileprefab = "player_cannon_focus_projectile"
	else
		damagemod = ATTACK.DAMAGE
		hitstun = ATTACK.HITSTUN
		pushback = ATTACK.PUSHBACK
		speed = ATTACK.SPEED
		range = ATTACK.RANGE
		projectileprefab = "player_cannon_projectile"
	end

	soundutil.PlayCodeSound(inst,fmodtable.Event.Cannon_shoot_light,{
		max_count = 1,
		fmodparams = {
			isFocusAttack = focus and 1 or 0,
			cannon_remainingAmmo_scaled = GetRemainingAmmo(inst) / GetMaxAmmo(inst),
			cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
		}
	})

	-- kill travel sound
	if inst.sg.mem.bullet and inst.sg.mem.bullet.handle then
		soundutil.KillSound(inst.sg.mem.bullet, inst.sg.mem.bullet.handle)
		inst.sg.mem.bullet = nil
	end

	-- Create the bullet, set it up, and position it correctly based on which state we're in.
	-- Neutral shot will have one y value, while airborne shot will have a different y value.
	local bullet = SGCommon.Fns.SpawnAtDist(inst, projectileprefab, 2)
	inst.sg.mem.bullet = bullet

	local pierce = false
	if inst.sg.mem.lightpierce then
		pierce = true
	elseif inst.sg.mem.lightfocuspierce and focus then
		pierce = true
	end

	bullet:Setup(inst, damagemod, hitstun, pushback, speed, range, focus, inst.sg.mem.attack_type, inst.sg.mem.attack_id, 1, 1, pierce)

	local bulletpos = bullet:GetPosition()
	local y_offset = inst.sg.statemem.projectile_y_offset ~= nil and inst.sg.statemem.projectile_y_offset or 1
	bullet.Transform:SetPosition(bulletpos.x, bulletpos.y + y_offset, bulletpos.z)

	-- Send an event for power purposes.
	inst:PushEvent("projectile_launched", { bullet })

	local previous_ammo = inst.sg.mem.ammo

	UpdateAmmo(inst, 1)

	if inst.sg.mem.ammo <= previous_ammo then
		PlayLowAmmoSound(inst, inst.sg.mem.ammo)
	end
end

local function DoShotFX(inst, current_ammo, fxname)
	-- Create particle FX, depending on how much ammo we have left.

	local fx_prefab = fxname --= current_ammo ~= nil and fxname.."_ammo1" or fxname

	if current_ammo then
		if inst.sg.mem.focus_sequence[current_ammo] then
			fx_prefab = fxname.."_ammo1_focus"
		else
			fx_prefab = fxname.."_ammo1_normal"
		end
	end
	-- local fx_prefab = current_ammo ~= nil and fxname.."_ammo"..current_ammo or fxname

	local fx = SGPlayerCommon.Fns.AttachSwipeFx(inst, fx_prefab, false, false)
	local power_fx = SGPlayerCommon.Fns.AttachPowerSwipeFx(inst, fxname, false, false) -- Don't use the ammo-adjusted name for the power prefab
end

local function DoMortarFX(inst, fxname, focus)
	-- Create particle FX, depending on how much ammo we have left.
	local fx_prefab = focus and fxname.."_focus" or fxname

	local fx = SGPlayerCommon.Fns.AttachSwipeFx(inst, fx_prefab, false, true)
	local power_fx = SGPlayerCommon.Fns.AttachPowerSwipeFx(inst, fxname, false, true) -- Don't use the focus-adjusted name for the power prefab
end

-- A blast shoots 5 bullets, and they should be delayed so they don't shoot in a boring pattern.
-- This pattern results in:
--[[
    o
o
  o
o
    o
]]
local delay_frames_per_blast_bullet =
{
	2,
	0,
	1,
	0,
	2,
}
-- Because the bullets come out at a different time, increase the range of the earlier shots so that they all die at the same time.
local extra_range_per_blast_bullet =
{
	0,
	1.25,
	1,
	1.25,
	0,
}

local function DoBlast(inst)
	-- First, initialize the attack
	inst:PushEvent("dodge")

	local ATTACK =  ATTACKS.BLAST

	local damagemod = ATTACK.DAMAGE
	local hitstun = ATTACK.HITSTUN
	local pushback = ATTACK.PUSHBACK
	local speed = ATTACK.SPEED
	local range = ATTACK.RANGE
	local projectileprefab = "player_cannon_shotgun_projectile"
	local remaining_ammo = GetRemainingAmmo(inst)

	-- If this is a focus attack, use FOCUS numbers. Otherwise, use default numbers.
	-- local focus = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)] == "lightattack" and true or false
	local focus = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)]
	if focus then
		damagemod = ATTACK.DAMAGE_FOCUS
		hitstun = ATTACK.HITSTUN_FOCUS
		pushback = ATTACK.PUSHBACK_FOCUS
		speed = ATTACK.SPEED_FOCUS
		range = ATTACK.RANGE_FOCUS
		projectileprefab = "player_cannon_shotgun_focus_projectile"
	end

	local numbullets
	local startangle
	local angleperbullet
	local bulletdelayframedata
	local bulletrangedata
	if inst.sg.mem.heavyblastmod then
		numbullets = inst.sg.mem.heavyblastmod.numbullets
		startangle = inst.sg.mem.heavyblastmod.startangle
		angleperbullet = inst.sg.mem.heavyblastmod.angleperbullet
		bulletdelayframedata = inst.sg.mem.heavyblastmod.delay_frames_per_blast_bullet
		bulletrangedata = inst.sg.mem.heavyblastmod.extra_range_per_blast_bullet
		if inst.sg.mem.heavyblastmod.damagemodmult then
			damagemod = damagemod * inst.sg.mem.heavyblastmod.damagemodmult
		end
	else
		numbullets = 5
		startangle = -30
		angleperbullet = 10
		bulletdelayframedata = delay_frames_per_blast_bullet
		bulletrangedata = extra_range_per_blast_bullet
	end

	-- Create 5 tiny bullets and spread them in a shotgun/spread pattern
	local bullets = {}
	for i=1,numbullets do
		local angle = startangle + (i * angleperbullet)
		local bullet = SGCommon.Fns.SpawnAtAngleDist(inst, projectileprefab, 2, angle)
		bullet:Hide()
		table.insert(bullets, bullet)

		local delay_frames = bulletdelayframedata[i]
		inst:DoTaskInAnimFrames(delay_frames, function()
			bullet:Show()

			local pierce = false
			if inst.sg.mem.heavypierce then
				pierce = true
			elseif inst.sg.mem.heavyfocuspierce and focus then
				pierce = true
			end

			bullet:Setup(inst, damagemod, hitstun, pushback, speed, range + bulletrangedata[i], focus, inst.sg.mem.attack_type, inst.sg.mem.attack_id, i, numbullets, pierce)

			local bulletpos = bullet:GetPosition()
			local y_offset = inst.sg.statemem.projectile_y_offset ~= nil and inst.sg.statemem.projectile_y_offset or 1.3
			bullet.Transform:SetPosition(bulletpos.x, bulletpos.y + y_offset, bulletpos.z)
		end)

	end

	inst:PushEvent("projectile_launched", bullets)

	local previous_ammo = inst.sg.mem.ammo
	
	UpdateAmmo(inst, 1)

	if inst.sg.mem.ammo <= previous_ammo then
		inst:DoTaskInAnimFrames(2, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
	end
end

local function DoMortar(inst, ammo)
	local facingright = inst.Transform:GetFacing() == FACING_RIGHT
	local num_bombs = ammo

	local ATTACK = ATTACKS.MORTAR

	local damagemod = ATTACK.DAMAGE
	local hitstun = ATTACK.HITSTUN
	local pushback = ATTACK.PUSHBACK
	local focus = inst.sg.mem.mortar_focus_sequence[GetRemainingAmmo(inst)] -- do not use "ammo" here, because we may be shooting 1 bullet while we have 6 left in the chamber
	local radius = ATTACK.RADIUS

	-- If this is a focus attack, use FOCUS numbers. Otherwise, use default numbers.
	if focus then
		damagemod = ATTACK.DAMAGE_FOCUS
		hitstun = ATTACK.HITSTUN_FOCUS
		pushback = ATTACK.PUSHBACK_FOCUS
		radius = ATTACK.RADIUS_FOCUS
	end

	-- Create 'num_bombs' mortars and arrange them in a star shape
	local bullets = {}

	local player = inst
	if num_bombs then
		inst.sg.mem.bombs_left_to_explode = num_bombs
	end

	for i = 1, num_bombs do
		local delay = MORTAR_SHOOT_TIMING[i]
		inst:DoTaskInAnimFrames(delay, function(inst)
			if inst ~= nil and inst:IsValid() then
				local bomb = SpawnPrefab("player_cannon_mortar_projectile", inst)

				-- Set the starting position of this mortar
				local offset = facingright and Vector3(1.5, 3, 0) or Vector3(-1.5, 3, 0) --inst.sg.statemem.right and Vector3(0, 0, 0) or Vector3(0, 0, 0)
				local x, z = inst.Transform:GetWorldXZ()
				bomb.Transform:SetPosition(x + offset.x, offset.y, z + offset.z)

				bomb:Setup(inst, damagemod, hitstun, radius, pushback, focus, "heavy_attack", i, num_bombs, inst.sg.mem.cannon_mortar_clusterbombs)
				-- Setup(owner, damage_mod, hitstun_animframes, hitboxradius, pushback, focus, attacktype, numberinbatch, maxinbatch, clusterbomb)

				-- Randomize the scale + rotation speed between a min/max for variance purposes
				local randomscale = krandom.Float(MORTAR_RANDOM_SCALE_MIN, MORTAR_RANDOM_SCALE_MAX)
				local randomrotationspeed = krandom.Float(MORTAR_RANDOM_ROTATESPEED_MIN, MORTAR_RANDOM_ROTATESPEED_MAX)
				bomb.AnimState:SetScale(randomscale, randomscale, randomscale)
				bomb.AnimState:SetDeltaTimeMultiplier(randomrotationspeed)
				bomb.owner = player
				bomb.num_bombs = num_bombs
				bomb.focus = focus
				bomb.sound_event = focus and fmodtable.Event.Cannon_Mortar_Explode_Counter_Focus or fmodtable.Event.Cannon_Mortar_Explode_Counter

				-- Set up the target
				local aim_x = inst.sg.mem.aim_root_x or 1
				local aim_z = inst.sg.mem.aim_root_z or 1
				local aim_offset = MORTAR_AIM_OFFSET[i]
				local offsetmult = facingright and -1 or 1
				local target_pos = Vector3(aim_x + aim_offset.x * offsetmult, 0, aim_z + aim_offset.z)

				table.insert(bullets, bomb)
				bomb:PushEvent("thrown", target_pos)
			end
		end)
	end

	inst:PushEvent("projectile_launched", bullets)
end

local function OnQuickriseHitBoxTriggered(inst, data)
	local ATTACK_DATA = ATTACKS.QUICKRISE

	for i = 1, #data.targets do
		local v = data.targets[i]

		local focushit = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)+1] -- Have to add +1 because ammo has already been updated by this point

		local hitstoplevel = focushit and ATTACK_DATA.HITSTOP_FOCUS or ATTACK_DATA.HITSTOP
		local damage_mod = focushit and ATTACK_DATA.DAMAGE_FOCUS or ATTACK_DATA.DAMAGE
		local pushback = focushit and ATTACK_DATA.PUSHBACK_FOCUS or ATTACK_DATA.PUSHBACK
		local hitstun = ATTACK_DATA.HITSTUN

		local dir = inst:GetAngleTo(v)

		local attack = Attack(inst, v)
		attack:SetDamageMod(damage_mod)
		attack:SetDir(dir)
		attack:SetHitstunAnimFrames(hitstun)
		attack:SetPushback(pushback)
		attack:SetFocus(focushit)
		attack:SetID(inst.sg.mem.attack_type)
		attack:SetNameID(inst.sg.mem.attack_id)

		local dist = inst:GetDistanceSqTo(v)
		dist = math.sqrt(dist)

		local force_knockdown = v.sg ~= nil and v.sg:HasStateTag("airborne")

		-- If really close to the center of the blast, do a knockdown. Otherwise, do a knockback.
		if dist <= ATTACK_DATA.RADIUS * 0.7 or force_knockdown then
			inst.components.combat:DoKnockdownAttack(attack)
		else
			inst.components.combat:DoKnockbackAttack(attack)
		end

		hitstoplevel = SGCommon.Fns.ApplyHitstop(attack, hitstoplevel)

		local hitfx_x_offset = 0
		local hitfx_y_offset = 0

		inst.components.combat:SpawnHitFxForPlayerAttack(attack, "hits_player_cannon_shot", v, inst, hitfx_x_offset, hitfx_y_offset, dir, hitstoplevel)

		-- TODO(combat): Why do we only spawn if target didn't block? We unconditionally spawn in hammer. Maybe we should move this to SpawnHitFxForPlayerAttack?
		if v.sg ~= nil and v.sg:HasStateTag("block") then
		else
			SpawnHurtFx(inst, v, hitfx_x_offset, dir, hitstoplevel)
		end
	end
end

local function OnShockwaveHitBoxTriggered(inst, data)
	assert(inst.sg.mem.attack_id == "SHOCKWAVE_WEAK" or inst.sg.mem.attack_id == "SHOCKWAVE_MEDIUM" or inst.sg.mem.attack_id == "SHOCKWAVE_STRONG", "Received wrong attack ID for shockwave attack.")
	local ATTACK_DATA = ATTACKS[inst.sg.mem.attack_id]

	for i = 1, #data.targets do
		local v = data.targets[i]

		local focushit = inst.sg.mem.focus_sequence[inst.sg.statemem.shockwave_ammo]

		local hitstoplevel = focushit and ATTACK_DATA.HITSTOP_FOCUS or ATTACK_DATA.HITSTOP
		local damage_mod = focushit and ATTACK_DATA.DAMAGE_FOCUS or ATTACK_DATA.DAMAGE
		local pushback = focushit and ATTACK_DATA.PUSHBACK_FOCUS or ATTACK_DATA.PUSHBACK
		local hitstun = ATTACK_DATA.HITSTUN

		local dir = inst:GetAngleTo(v)

		local attack = Attack(inst, v)
		attack:SetDamageMod(damage_mod)
		attack:SetDir(dir)
		attack:SetHitstunAnimFrames(hitstun)
		attack:SetPushback(pushback)
		attack:SetFocus(focushit)
		attack:SetID(inst.sg.mem.attack_type)
		attack:SetNameID(inst.sg.mem.attack_id)

		local dist = inst:GetDistanceSqTo(v)
		dist = math.sqrt(dist)

		local force_knockdown = v.sg ~= nil and v.sg:HasStateTag("airborne")

		local hit_v

		-- If really close to the center of the blast, do a knockdown. Otherwise, do a knockback.
		if ATTACK_DATA.KNOCKDOWN and dist <= ATTACK_DATA.RADIUS * ATTACK_DATA.KNOCKDOWN_RADIUS or force_knockdown then
			hit_v = inst.components.combat:DoKnockdownAttack(attack)
		else
			hit_v = inst.components.combat:DoKnockbackAttack(attack)
		end

		if hit_v then
			hitstoplevel = SGCommon.Fns.ApplyHitstop(attack, hitstoplevel)

			local hitfx_x_offset = 0
			local hitfx_y_offset = 0

			inst.components.combat:SpawnHitFxForPlayerAttack(attack, "hits_player_cannon_shot", v, inst, hitfx_x_offset, hitfx_y_offset, dir, hitstoplevel)

			-- TODO(combat): Why do we only spawn if target didn't block? We unconditionally spawn in hammer. Maybe we should move this to SpawnHitFxForPlayerAttack?
			if v.sg ~= nil and v.sg:HasStateTag("block") then
			else
				SpawnHurtFx(inst, v, hitfx_x_offset, dir, hitstoplevel)
			end
		end
	end
end

local function DoShockwaveSelfAttack(inst)
	if TheWorld:HasTag("town") then
		-- Don't self-damage in town
		return
	end

	assert(inst.sg.mem.attack_id == "SHOCKWAVE_WEAK" or inst.sg.mem.attack_id == "SHOCKWAVE_MEDIUM" or inst.sg.mem.attack_id == "SHOCKWAVE_STRONG", "Received wrong attack ID for shockwave attack.")
	local ATTACK_DATA = ATTACKS[inst.sg.mem.attack_id]

	if ATTACK_DATA.SELF_DAMAGE <= 0 then
		return
	end

	local focushit = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)]

	local hitstoplevel = focushit and ATTACK_DATA.HITSTOP_FOCUS or ATTACK_DATA.HITSTOP
	local pushback = focushit and ATTACK_DATA.PUSHBACK_FOCUS or ATTACK_DATA.PUSHBACK
	local hitstun = ATTACK_DATA.HITSTUN

	local dir = inst:GetAngleTo(inst)

	local attack = Attack(inst, inst)
	attack:SetOverrideDamage(ATTACK_DATA.SELF_DAMAGE)
	attack:SetHitstunAnimFrames(hitstun)
	attack:SetPushback(pushback)
	attack:SetFocus(focushit)
	attack:SetID(inst.sg.mem.attack_type)
	attack:SetNameID(inst.sg.mem.attack_id)

	inst.components.combat:DoBasicAttack(attack)
	-- Flicker red
	SGCommon.Fns.BlinkAndFadeColor(inst, { 255/255, 50/255, 50/255, 1 }, 8)
end

local function OnBackfireHitBoxTriggered(inst, data)
	-- This is for the first burst in the backfire strong, right after blasting.
	-- assert(inst.sg.mem.attack_id == "BACKFIRE_STRONG_EARLY" or inst.sg.mem.attack_id == "BACKFIRE_STRONG_LATE", "Received wrong attack ID for BACKFIRE attack. Maybe some timing issue?")
	local ATTACK_DATA = ATTACKS[inst.sg.mem.attack_id]

	-- if inst.sg.mem.backfire_sound then
	-- 	soundutil.KillSound(inst, inst.sg.mem.backfire_sound)
	-- 	inst.sg.mem.backfire_sound = nil
	-- end

	for i = 1, #data.targets do
		local v = data.targets[i]

		local focushit = inst.sg.mem.focus_sequence[inst.sg.statemem.backfire_ammo]

		local hitstoplevel = focushit and ATTACK_DATA.HITSTOP_FOCUS or ATTACK_DATA.HITSTOP
		local damage_mod = focushit and ATTACK_DATA.DAMAGE_FOCUS or ATTACK_DATA.DAMAGE
		local pushback = focushit and ATTACK_DATA.PUSHBACK_FOCUS or ATTACK_DATA.PUSHBACK
		local hitstun = ATTACK_DATA.HITSTUN

		local dir = inst:GetAngleTo(v)

		local attack = Attack(inst, v)
		attack:SetDamageMod(damage_mod)
		attack:SetDir(dir)
		attack:SetHitstunAnimFrames(hitstun)
		attack:SetPushback(pushback)
		attack:SetFocus(focushit)
		attack:SetID(inst.sg.mem.attack_type)
		attack:SetNameID(inst.sg.mem.attack_id)

		if inst.sg.statemem.hitflags then
			attack:SetHitFlags(inst.sg.statemem.hitflags)
		end

		local force_knockdown = v.sg ~= nil and v.sg:HasStateTag("airborne")

		local hit_v
		if ATTACK_DATA.KNOCK == "KNOCKDOWN" or force_knockdown then
			hit_v = inst.components.combat:DoKnockdownAttack(attack)
		elseif ATTACK_DATA.KNOCK == "KNOCKBACK" then
			hit_v = inst.components.combat:DoKnockbackAttack(attack)
		else
			hit_v = inst.components.combat:DoBasicAttack(attack)
		end

		if hit_v then
			hitstoplevel = SGCommon.Fns.ApplyHitstop(attack, hitstoplevel, { disable_self_hitstop = true }) -- Player will be flying through the air, so don't pause.

			local hitfx_x_offset = inst.sg.statemem.hitfx_x_offset or 0
			local hitfx_y_offset = inst.sg.statemem.hitfx_y_offset or 0

			inst.components.combat:SpawnHitFxForPlayerAttack(attack, "hits_player_cannon_shot", v, inst, hitfx_x_offset, hitfx_y_offset, dir, hitstoplevel)

			-- TODO(combat): Why do we only spawn if target didn't block? We unconditionally spawn in hammer. Maybe we should move this to SpawnHitFxForPlayerAttack?
			if v.sg ~= nil and v.sg:HasStateTag("block") then
			else
				SpawnHurtFx(inst, v, hitfx_x_offset, dir, hitstoplevel)
			end
		end
	end
end

local function DoBackfireSelfAttack(inst)
	if TheWorld:HasTag("town") then
		-- Don't self-damage in town
		return
	end

	assert(inst.sg.mem.attack_id == "BACKFIRE_STRONG_EARLY" or inst.sg.mem.attack_id == "BACKFIRE_STRONG_LATE", "Received wrong attack ID for BACKFIRE attack. Maybe some timing issue?")
	local ATTACK_DATA = ATTACKS[inst.sg.mem.attack_id]

	if ATTACK_DATA.SELF_DAMAGE <= 0 then
		return
	end

	local focushit = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)]

	local hitstoplevel = focushit and ATTACK_DATA.HITSTOP_FOCUS or ATTACK_DATA.HITSTOP
	local pushback = focushit and ATTACK_DATA.PUSHBACK_FOCUS or ATTACK_DATA.PUSHBACK
	local hitstun = ATTACK_DATA.HITSTUN

	local dir = inst:GetAngleTo(inst)

	local attack = Attack(inst, inst)
	attack:SetOverrideDamage(ATTACK_DATA.SELF_DAMAGE)
	attack:SetHitstunAnimFrames(hitstun)
	attack:SetPushback(pushback)
	attack:SetFocus(focushit)
	attack:SetID(inst.sg.mem.attack_type)
	attack:SetNameID(inst.sg.mem.attack_id)

	inst.components.combat:DoBasicAttack(attack)

	hitstoplevel = SGCommon.Fns.ApplyHitstop(attack, hitstoplevel)

	local hitfx_x_offset = ATTACK_DATA.SELF_HIT_FX_X_OFFSET
	local hitfx_y_offset = ATTACK_DATA.SELF_HIT_FX_Y_OFFSET

	inst.components.combat:SpawnHitFxForPlayerAttack(attack, "hits_player_cannon_shot", inst, inst, hitfx_x_offset, hitfx_y_offset, inst, hitstoplevel)
	-- Flicker red
	SGCommon.Fns.BlinkAndFadeColor(inst, { 255/255, 50/255, 50/255, 1 }, 8)

	-- TODO(combat): Why do we only spawn if target didn't block? We unconditionally spawn in hammer. Maybe we should move this to SpawnHitFxForPlayerAttack?
	if inst.sg ~= nil and inst.sg:HasStateTag("block") then
	else
		SpawnHurtFx(inst, inst, hitfx_x_offset, dir, hitstoplevel)
	end
end

local function GetReloadState(inst)
	-- Reload
	local nextstate
	if inst.sg.statemem.perfectwindow then
		nextstate = "planted_reload_fast"
	elseif inst.sg.statemem.earlywindow then
		nextstate = "planted_reload_slow_early"
	else
		nextstate = "planted_reload_slow_late"
	end
	return nextstate
end

local function GetShockwavePreState(inst)
	-- Reload
	local nextstate
	if inst.sg.statemem.perfectwindow then
		nextstate = "shockwave_ammocheck"
	elseif inst.sg.statemem.earlywindow then
		nextstate = "shockwave_pre_early"
	end
	return nextstate
end

local function GetBackfirePreState(inst)
	-- Reload
	local nextstate
	if inst.sg.statemem.perfectwindow then
		nextstate = "backfire_ammocheck"
	elseif inst.sg.statemem.earlywindow and GetRemainingAmmo(inst) > 0 then
		-- This animation hops onto the cannon, which would then pop as we transition back to the "chk-chk!" no ammo state.
		-- So we must check at this point if we have enough ammo. If not, go right to ammo_check so we don't hop onto the cannon.
		nextstate = "backfire_pre_early"
	elseif GetRemainingAmmo(inst) <= 0 then
		nextstate = "backfire_pre_early_noammo"
	end
	return nextstate
end

local function GetMortarPreState(inst)
	-- Reload
	local nextstate
	if inst.sg.statemem.perfectwindow then
		nextstate = "mortar_ammocheck"
	elseif inst.sg.statemem.earlywindow then
		nextstate = "mortar_pre_early"
	end
	return nextstate
end

-- Default data for tools functions
local function BlastToStatesDataForTools(inst)
	return {
		maxspeed = -TUNING.GEAR.WEAPONS.CANNON.ROLL_VELOCITY,
		speed = 0,
		framessliding = 0,
	}
end

local function CheckForHeavyQuickRise(inst, data)
	if data.control == "heavyattack" and inst.sg.statemem.canheavydodgespecial then

		if GetRemainingAmmo(inst) > 0 then
			inst.sg:GoToState("cannon_quickrise", true)
		else
			inst.sg:GoToState("cannon_quickrise_noammo", true)
		end

		return true
	end

	return false
end

local function ConfigureEquipmentStats(inst)
	local equipped_weapon = inst.components.inventoryhoard:GetEquippedItem(Equipment.Slots.WEAPON)

	inst.sg.mem.ammo_max = equipped_weapon and equipped_weapon.stats and equipped_weapon.stats.AMMO
	inst.sg.mem.ammo = inst.sg.mem.ammo_max

	inst.sg.mem.focus_sequence = equipped_weapon and equipped_weapon.stats and equipped_weapon.stats.FOCUS_SEQUENCE
	inst.sg.mem.mortar_focus_sequence = equipped_weapon and equipped_weapon.stats and equipped_weapon.stats.MORTAR_FOCUS_SEQUENCE

	inst.sg.mem.heavydodge = true -- For quickrising

	UpdateAmmoSymbols(inst)
end

local events =
{
	EventHandler("cannon_reload", function(inst, amount)
		OnReload(inst, amount)
	end),

	EventHandler("loadout_changed", function(inst, data)
		ConfigureEquipmentStats(inst)
	end),
	EventHandler("controlevent", function(inst, data)
		CheckForHeavyQuickRise(inst, data)
	end),
	EventHandler("skill_hit", function(inst)
		-- Reload one ammo whenever the Skill hits.
		if inst.sg.mem.ammo < inst.sg.mem.ammo_max then
			inst:PushEvent("cannon_butt_reload")
			inst:PushEvent("cannon_reload", 1)
			soundutil.PlayCodeSound(inst, fmodtable.Event.Skill_Cannon_ReloadAmmo, {
						max_count = 1,
						fmodparams = {
							cannon_remainingAmmo_scaled = inst.sg.mem.ammo / inst.sg.mem.ammo_max,
							cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
						}
					})

			powerutil.SpawnParticlesAtPosition(inst:GetPosition(), "cannon_skill_recharge", 1, inst)
		end
	end),
}
SGPlayerCommon.Events.AddAllBasicEvents(events)

local states =
{
	State({
		name = "init",
		onenter = function(inst)
			ConfigureEquipmentStats(inst)

			inst.sg:GoToState("idle")
		end,
	}),

	State({
		name = "default_light_attack",
		onenter = function(inst, transitiondata)
			if GetRemainingAmmo(inst) > 0 then
				inst.sg:GoToState("shoot", transitiondata ~= nil and transitiondata or nil)
			else
				inst.sg:GoToState("shoot_noammo")
			end
		end,
	}),

	State({
		name = "default_heavy_attack",
		onenter = function(inst, transitiondata)
			if GetRemainingAmmo(inst) > 0 then
				inst.sg:GoToState("blast", transitiondata ~= nil and transitiondata or nil)
			else
				inst.sg:GoToState("shoot_noammo")
			end
		end,
	}),

	State({
		name = "default_dodge",
		onenter = function(inst)
			--NOTE: If we want to actually animate a "knockdown -> cannon_plant" sequence, use this logic to go to a specific state.
			-- if table.contains(inst.sg.laststate.tags, "knockdown") then
			-- 	inst.sg:GoToState("blast_TO_plant")
			-- else
			-- 	inst.sg:GoToState("cannon_plant_pre")
			-- end

			inst.sg:GoToState("cannon_plant_pre")
		end,
	}),

	State({
		name = "shoot",
		tags = { "busy", "light_attack" },

		default_data_for_tools = function(inst)
			return { attack_type = "light_attack" }
		end,

		onenter = function(inst, transitiondata)
			-- transitiondata =
			-- exists if we are canceling into this state from a blast-dodge
			-- {
			-- 	maxspeed = the overall speed of this blast backwards
			-- 	speed = the speed we are currently moving
			-- 	framessliding = how many frames we have already been sliding, so we can continue the same slide
			-- }
			inst.AnimState:PlayAnimation("cannon_atk1")
			inst.Network:FlushAllHistory()	-- Make sure this anim 'skips' the network buffered anim history

			inst.sg.mem.attack_id = "SHOOT"
			inst.sg.mem.attack_type = "light_attack"

			ParticleSystemHelper.MakeEventSpawnParticles(inst, {
				duration=60.0,
				offx=0.9,
				offy=0.9,
				offz=0.0,
				particlefxname= inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)] and "cannon_shot_focus" or "cannon_shot",
				use_entity_facing=true
			})

			DoShotFX(inst, GetRemainingAmmo(inst), "fx_player_cannon_atk1")
			DoShoot(inst)
			-- CheckIfDodging(inst, transitiondata) -- If we canceled into this shot from another blast

			inst.sg.statemem.weightmult = GetWeightVelocityMult(inst)
			inst:PushEvent("attack_state_start")
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst)
		end,

		timeline =
		{
			--physics
			FrameEvent(0, DoShootKickback),

			FrameEvent(1, function(inst) inst.Physics:SetMotorVel(-3 * inst.sg.statemem.weightmult) end),
			FrameEvent(3, function(inst) inst.Physics:SetMotorVel(-2 * inst.sg.statemem.weightmult) end),
			FrameEvent(4, function(inst) inst.Physics:SetMotorVel(-1 * inst.sg.statemem.weightmult) end),
			FrameEvent(5, function(inst) inst.Physics:SetMotorVel(0 * inst.sg.statemem.weightmult) end),

			FrameEvent(7, SGPlayerCommon.Fns.RemoveBusyState),
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
				inst.Physics:Stop()
			end),
		},
	}),

	State({
		name = "blast",
		tags = { "busy", "airborne", "heavy_attack", "dodge", "dodging_backwards" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_H_atk")
			inst:PushEvent("attack_state_start")

			inst.sg.mem.attack_id = "BLAST"
			inst.sg.mem.attack_type = "heavy_attack"

			inst.sg.statemem.fx_x_offset = 0.9
			inst.sg.statemem.fx_y_offset = 0.91

			local focus = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)]

			soundutil.PlayCodeSound(inst,fmodtable.Event.Cannon_shoot_heavy,{
				max_count = 1,
				fmodparams = {
					isFocusAttack = focus and 1 or 0,
					cannon_heavyShotType = 0,
					cannon_remainingAmmo_scaled = GetRemainingAmmo(inst) / GetMaxAmmo(inst),
					cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param
				}
			})
			--fx
			ParticleSystemHelper.MakeEventSpawnParticles(inst, {
				duration=45.0,
				offx=0.87,
				offy=1.31,
				offz=0.0,
				particlefxname= focus and "cannon_shot_wide_focus" or "cannon_shot_wide",
				use_entity_facing=true,
			})

			DoShotFX(inst, GetRemainingAmmo(inst), "fx_player_cannon_h_atk")
			DoBlast(inst)
			ConfigureNewDodge(inst)

			inst.Physics:StartPassingThroughObjects()
			SGCommon.Fns.StartJumpingOverHoles(inst)
			SGPlayerCommon.Fns.SetRollPhysicsSize(inst)
			inst.HitBox:SetInvincible(true)
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst)
		end,

		timeline =
		{
			FrameEvent(1, function(inst)
				DoBlastKickback(inst) -- Start with a strong burst
				StartNewDodge(inst)
			end),

			--CANCELS
			FrameEvent(5, function(inst)
				inst.sg.statemem.canshoot = true
				SGPlayerCommon.Fns.DetachSwipeFx(inst)
				SGPlayerCommon.Fns.DetachPowerSwipeFx(inst)
			end),
			FrameEvent(7, function(inst)
				inst.sg.statemem.canplant = true
				if inst.sg.statemem.triedplantearly then
					local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
					inst.sg:GoToState("blast_TO_plant", transitiondata)
				end
			end),
		},

		onexit = function(inst)
			inst.HitBox:SetInvincible(false)
			SGPlayerCommon.Fns.UndoRollPhysicsSize(inst)
			SGCommon.Fns.StopJumpingOverHoles(inst)
			SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst)
		end,

		events =
		{
			EventHandler("controlevent", function(inst, data)
				local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
				if inst.sg.statemem.canshoot then
					if data.control == "heavyattack" then
						SGCommon.Fns.FaceActionTarget(inst, data, true, true)
						if GetRemainingAmmo(inst) > 0 then
							inst.sg:GoToState("blast_TO_airblast", transitiondata) -- JAN14 switch to double-dodge state
						else
							inst.sg:GoToState("blast_TO_noammo", transitiondata)
						end
					elseif data.control == "lightattack" then
						SGCommon.Fns.FaceActionTarget(inst, data, true, true)
						if GetRemainingAmmo(inst) > 0 then
							inst.sg:GoToState("blast_TO_airshoot", transitiondata)
						else
							inst.sg:GoToState("blast_TO_noammo", transitiondata)
						end
					end
				end

				if data.control == "dodge" then
					if inst.sg.statemem.canplant then
						inst.sg:GoToState("blast_TO_plant", transitiondata)
					else
						inst.sg.statemem.triedplantearly = true
					end
				end
			end),

			EventHandler("animover", function(inst)
				local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
				inst.sg:GoToState("blast_TO_land", transitiondata)
			end),
		},
	}),

	State({
		name = "blast_TO_noammo",
		tags = { "busy", "airborne" },

		onenter = function(inst, transitiondata)
			inst.AnimState:PlayAnimation("cannon_H_to_L_atk")
			-- "Dodge" stuff, using transitiondata
			CheckIfDodging(inst, transitiondata)

		end,

		onupdate = function(inst)
			DoDodgeMovement(inst)
		end,

		timeline =
		{
			FrameEvent(1, function(inst)
				PlayNoAmmoSound(inst)
			end),
		},

		onexit = function(inst)
			inst.HitBox:SetInvincible(false)
			SGPlayerCommon.Fns.UndoRollPhysicsSize(inst)
			SGCommon.Fns.StopJumpingOverHoles(inst)
		end,


		events =
		{
			EventHandler("animover", function(inst)
				local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
				inst.sg:GoToState("blast_TO_land", transitiondata)
			end),
		},
	}),

	State({
		name = "blast_TO_land",
		tags = { "busy" },

		default_data_for_tools = BlastToStatesDataForTools,

		onenter = function(inst, transitiondata)
			inst.AnimState:PlayAnimation("cannon_H_land")

			-- "Dodge" stuff, using transitiondata
			CheckIfDodging(inst, transitiondata)
			--

			inst.sg.statemem.canplant = true
			inst.sg.statemem.canslidingact = true
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst)
		end,

		timeline =
		{
			--CANCELS
			FrameEvent(2, SGPlayerCommon.Fns.SetCanSkill),
		},

		onexit = function(inst)
			inst.HitBox:SetInvincible(false)
			inst.Physics:Stop()
		end,


		events =
		{
			EventHandler("controlevent", function(inst, data)
				local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
				if inst.sg.statemem.canslidingact then
					if data.control == "heavyattack" then
						inst.sg:GoToState("default_heavy_attack", transitiondata)
					elseif data.control == "lightattack" then
						inst.sg:GoToState("default_light_attack", transitiondata)
					end
				end

				if inst.sg.statemem.canplant then
					if data.control == "dodge" then
						inst.sg:GoToState("cannon_plant_pre", transitiondata) -- JAN14 switch to midair-plant state
					end
				end
			end),

			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "blast_TO_airshoot",
		tags = { "busy", "airborne", "light_attack" },

		default_data_for_tools = BlastToStatesDataForTools,

		onenter = function(inst, transitiondata)
			inst.AnimState:PlayAnimation("cannon_H_to_L_atk")
			inst.sg.mem.attack_id = "BLACK_TO_AIRSHOOT"
			inst.sg.mem.attack_type = "light_attack"
			inst:PushEvent("attack_state_start")
			inst.sg.statemem.projectile_y_offset = 1.5

			SGCommon.Fns.StartJumpingOverHoles(inst)
			inst.Physics:StartPassingThroughObjects()

			-- Dodge stuff, using transitiondata
			CheckIfDodging(inst, transitiondata) -- If we were already dodging, continue that dodge's momentum.

			-- Mid-air shooting presentation
			if GetRemainingAmmo(inst) > 0 then
				local hitstop = HitStopLevel.LIGHT
				inst.components.hitstopper:PushHitStop(hitstop)
				inst.sg.statemem.pausedodgemovement = true -- Pause dodge movement momentarily to let the eye register the shot

				inst.sg.statemem.afterhitstop_task = inst:DoTaskInAnimFrames(hitstop+1, function()
					if inst ~= nil and inst:IsValid() then
						if GetRemainingAmmo(inst) > 0 then
							inst.sg.statemem.pausedodgemovement = false
							inst.sg.statemem.framessliding = math.floor(inst.sg.statemem.framessliding / 2) -- Regain some momentum

							DoShootKickback(inst)
							DoAirShootMovement(inst)
							DoShoot(inst)
						end
					end
				end)
			else
				inst.sg.statemem.cantshoot = true
			end
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst)
		end,

		timeline =
		{
			-- FX: Because this one state needs to be reused for both "ammo" and "noammo" versions, we will play the FX here.
			FrameEvent(1, function(inst)
				if GetRemainingAmmo(inst) + 1 > 0 then
					-- Normal shot
					-- No ammo version is below in the controlevent EventHandler
					ParticleSystemHelper.MakeEventSpawnParticles(inst, {
						duration=45.0,
						offx=0.5,
						offy=2.0,
						offz=0.0,
						particlefxname= inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)+1] and "cannon_shot_focus" or "cannon_shot",
						use_entity_facing=true
					})
					DoShotFX(inst, GetRemainingAmmo(inst)+1, "fx_player_cannon_h_to_l")
				end
			end),

			FrameEvent(7, function(inst)
				SGPlayerCommon.Fns.DetachSwipeFx(inst)
				SGPlayerCommon.Fns.DetachPowerSwipeFx(inst)
			end),

			--CANCELS
			FrameEvent(7, function(inst)
				inst.sg.statemem.canshoot = true
				inst.sg.statemem.airplant = true
				if inst.sg.statemem.triedplantearly then
					local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
					inst.sg:GoToState("blast_TO_plant", transitiondata)
				end
			end),
			FrameEvent(8, function(inst)
				inst.sg.statemem.airplant = false
				inst.sg.statemem.groundplant = true
			end),
		},

		onexit = function(inst)
			inst.HitBox:SetInvincible(false)
			inst.Physics:Stop()
			SGCommon.Fns.StopJumpingOverHoles(inst)
			SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst)
			if inst.sg.statemem.afterhitstop_task then
				inst.sg.statemem.afterhitstop_task:Cancel()
				inst.sg.statemem.afterhitstop_task = nil
			end
		end,


		events =
		{
			EventHandler("controlevent", function(inst, data)
				if inst.sg.statemem.canshoot then
					local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
					if data.control == "heavyattack" and not inst.sg.statemem.cantshoot then
						SGCommon.Fns.FaceActionTarget(inst, data, true, true)
						if GetRemainingAmmo(inst) > 0 then
							inst.sg:GoToState("blast_TO_airblast", transitiondata)
						else
							inst.sg:GoToState("blast_TO_noammo", transitiondata)
						end
					elseif data.control == "lightattack" and not inst.sg.statemem.cantshoot then
						SGCommon.Fns.FaceActionTarget(inst, data, true, true)
						if GetRemainingAmmo(inst) > 0 then
							inst.sg:GoToState("blast_TO_airshoot", transitiondata)
						else
							inst.sg:GoToState("blast_TO_noammo", transitiondata)
						end
					end
				end

				if data.control == "dodge" then
					if inst.sg.statemem.airplant then
						local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
						inst.sg:GoToState("blast_TO_plant", transitiondata)
					elseif inst.sg.statemem.groundplant then
						local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
						inst.sg:GoToState("cannon_plant_pre", transitiondata)
					else
						inst.sg.statemem.triedplantearly = true
					end
				end
			end),

			EventHandler("animover", function(inst)
				local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
				inst.sg:GoToState("blast_TO_land", transitiondata)
			end),
		},
	}),

	State({
		name = "airshoot_TO_airshoot",
		tags = { "busy", "light_attack" },

		default_data_for_tools = BlastToStatesDataForTools,

		onenter = function(inst, transitiondata)
			inst.AnimState:PlayAnimation("cannon_H_to_L_atk")
			inst.sg.mem.attack_id = "AIRSHOOT_TO_AIRSHOOT"
			inst.sg.mem.attack_type = "light_attack"
			inst:PushEvent("attack_state_start")
			inst.sg.statemem.projectile_y_offset = 1.4

			SGCommon.Fns.StartJumpingOverHoles(inst)

			-- "Dodge" stuff:
			CheckIfDodging(inst, transitiondata)
			--

			-- Mid-air shooting presentation
			if GetRemainingAmmo(inst) > 0 then
				local hitstop = HitStopLevel.LIGHT
				inst.components.hitstopper:PushHitStop(hitstop)
				inst.sg.statemem.pausedodgemovement = true -- Pause dodge movement momentarily to let the eye register the shot

				inst.sg.statemem.afterhitstop_task = inst:DoTaskInAnimFrames(hitstop+1, function()
					if inst ~= nil and inst:IsValid() then
						if GetRemainingAmmo(inst) > 0 then
							DoShootKickback(inst)
							inst.sg.statemem.framessliding = math.floor(inst.sg.statemem.framessliding / 2)
							inst.sg.statemem.pausedodgemovement = false
							DoShoot(inst)
							DoShotFX(inst, GetRemainingAmmo(inst)+1, "fx_player_cannon_h_to_l")

							local focus = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)+1]
							ParticleSystemHelper.MakeEventSpawnParticles(inst, {
								duration=45.0,
								offx=0.5,
								offy=2.0,
								offz=0.0,
								particlefxname= focus and "cannon_shot_focus" or "cannon_shot",
								use_entity_facing=true,
							})
						end
					end
				end)
			end
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst)
		end,

		timeline =
		{
			--CANCELS
			-- FrameEvent(2, function(inst) inst.sg.statemem.canshoot = true end),
		},

		onexit = function(inst)
			inst.HitBox:SetInvincible(false)
			inst.Physics:Stop()
			SGCommon.Fns.StopJumpingOverHoles(inst)
			if inst.sg.statemem.afterhitstop_task then
				inst.sg.statemem.afterhitstop_task:Cancel()
				inst.sg.statemem.afterhitstop_task = nil
			end
		end,


		events =
		{
			EventHandler("controlevent", function(inst, data)
				-- if inst.sg.statemem.canslidingact then
				-- 	local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
				-- 	if data.control == "heavyattack" then
				-- 		inst.sg:GoToState("default_heavy_attack", transitiondata)
				-- 	elseif data.control == "lightattack" then
				-- 		inst.sg:GoToState("default_light_attack", transitiondata)
				-- 	elseif data.control == "dodge" then
				-- 		inst.sg:GoToState("cannon_plant_pre", transitiondata)
				-- 	end
				-- end
			end),

			EventHandler("animover", function(inst)
				inst.sg:GoToState("blast_TO_land")
			end),
		},
	}),

	State({
		name = "blast_TO_airblast",
		tags = { "busy", "airborne", "airborne_high", "heavy_attack", "dodge", "dodging_backwards" },

		onenter = function(inst, transitiondata)
			inst.AnimState:PlayAnimation("cannon_H_to_H_atk")
			inst.sg.mem.attack_id = "BLAST_TO_AIRBLAST"
			inst.sg.mem.attack_type = "heavy_attack"
			inst:PushEvent("attack_state_start")

			inst.sg.statemem.projectile_y_offset = 1.7

			SGCommon.Fns.StartJumpingOverHoles(inst)

			-- "Dodge" stuff:
			CheckIfDodging(inst, transitiondata) -- If we were already dodging, continue that dodge's momentum.
			--

			-- Mid-air shooting presentation
			if GetRemainingAmmo(inst) > 0 then
				local hitstop = HitStopLevel.MEDIUM
				inst.components.hitstopper:PushHitStop(hitstop)
				inst.sg.statemem.pausedodgemovement = true -- Pause dodge movement momentarily to let the eye register the shot
				inst:DoTaskInAnimFrames(hitstop, function()
					if inst ~= nil and inst:IsValid() then
						if GetRemainingAmmo(inst) > 0 then
							inst.Physics:StartPassingThroughObjects()

							inst.sg.statemem.pausedodgemovement = false
							ConfigureNewDodge(inst)
							StartNewDodge(inst)
							SGPlayerCommon.Fns.SetRollPhysicsSize(inst)

							inst.AnimState:SetFrame(1) -- Force us to jump to the post-"hitstop" freezeframe

							local focus = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)]

							--fx
							ParticleSystemHelper.MakeEventSpawnParticles(inst, {
								duration=45.0,
								offx=0.8,
								offy=1.6,
								offz=0.0,
								particlefxname= focus and "cannon_shot_wide_focus" or "cannon_shot_wide",
								use_entity_facing=true,
							})
							ParticleSystemHelper.MakeEventSpawnParticles(inst, {
								detachatexitstate=true,
								duration=15.0,
								followsymbol="swap_fx",
								ischild=true,
								offx=0.0,
								offy=0.0,
								offz=0.0,
								particlefxname= focus and "cannon_backflip_trail_focus" or "cannon_backflip_trail",
								use_entity_facing=true,
							})

							-- sound
							soundutil.PlayCodeSound(inst,fmodtable.Event.Cannon_shoot_blast,{
								max_count = 1,
								fmodparams = {
									isFocusAttack = focus and 1 or 0,
									cannon_heavyShotType = 1,
									cannon_remainingAmmo_scaled = GetRemainingAmmo(inst) / GetMaxAmmo(inst),
									cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param
								}
							})

							DoAirBlastKickback(inst)
							DoShotFX(inst, GetRemainingAmmo(inst), "fx_player_cannon_h_to_h")
							DoBlast(inst)

						end
					end
				end)
			end
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst)
		end,

		timeline =
		{
			-- Movement + state tags
			FrameEvent(13, function(inst)
				inst.sg:RemoveStateTag("airborne_high")

			end),

			FrameEvent(16, function(inst)
				inst.sg.statemem.speed = 0
				inst.sg:RemoveStateTag("airborne")
				SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst)
				SGCommon.Fns.StopJumpingOverHoles(inst)
			end),

			--CANCELS
			FrameEvent(12, function(inst)
				inst.sg.statemem.airplant = true
				inst.sg.statemem.canquickrise = true
				if inst.sg.statemem.triedplantearly then
					local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
					inst.sg:GoToState("blast_TO_plant", transitiondata)
				end
			end),
			FrameEvent(15, function(inst)
				inst.sg.statemem.airplant = false
				inst.sg.statemem.groundplant = true
			end),
			FrameEvent(16, function(inst)
				inst.sg.statemem.canquickrise = false
				inst.sg.statemem.heavycombostate = nil
			end),
			FrameEvent(17, function(inst)
				inst.sg.statemem.heavycombostate = "default_heavy_attack"
				SGPlayerCommon.Fns.TryQueuedAction(inst, "heavyattack")
			end),
			FrameEvent(24, SGPlayerCommon.Fns.SetCanAttackOrAbility),
			FrameEvent(30, SGPlayerCommon.Fns.RemoveBusyState),
		},

		onexit = function(inst)
			inst.HitBox:SetInvincible(false)
			SGPlayerCommon.Fns.UndoRollPhysicsSize(inst)
			SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst)
			inst.Physics:Stop()
			SGCommon.Fns.StopJumpingOverHoles(inst)
		end,


		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),

			EventHandler("controlevent", function(inst, data)
				if data.control == "dodge" then
					local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
					if inst.sg.statemem.airplant then
						inst.sg:GoToState("blast_TO_plant", transitiondata)
					elseif inst.sg.statemem.groundplant then
						inst.sg:GoToState("cannon_plant_pre", transitiondata)
					else
						inst.sg.statemem.triedplantearly = true
					end
				elseif data.control == "heavyattack" then
					if inst.sg.statemem.canquickrise and GetRemainingAmmo(inst) > 0 then
						SGCommon.Fns.FaceActionTarget(inst, data, true, true)
						inst.sg:GoToState("cannon_quickrise")
					end
				end
			end),
		},
	}),

	State({
		name = "shoot_noammo",
		tags = { "busy" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_empty")

			PlayNoAmmoSound(inst)

			inst.Physics:Stop()
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "cannon_plant_pre",
		tags = { "busy" },

		onenter = function(inst, transitiondata)
			-- transitiondata =
			-- {
			-- 	maxspeed = the overall speed of this blast backwards
			-- 	speed = the speed we are currently moving
			-- 	framessliding = how many frames we have already been sliding, so we can continue the same slide
			-- }
			inst.AnimState:PlayAnimation("cannon_atk2_pre")
			inst.sg.statemem.perfectwindow = false
			inst.sg.statemem.earlywindow = true

			-- "Dodge" stuff:
			CheckIfDodging(inst, transitiondata)
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst, true)
		end,

		timeline =
		{
			FrameEvent(5, function(inst)
				-- SGCommon.Fns.BlinkAndFadeColor(inst, {0.25, 0.25, 0.25}, 6) -- 2f eligibility here, 4f eligibility in the hold frame
				inst.sg.statemem.earlywindow = false
				inst.sg.statemem.perfectwindow = true
			end),
		},

		onexit = function(inst)
			inst.Physics:Stop()
		end,

		events =
		{
			EventHandler("controlevent", function(inst, data)
				local nextstate
				local transitiondata
				if data.control == "dodge" then
					nextstate = GetReloadState(inst)
				elseif data.control == "lightattack" then
					nextstate = GetShockwavePreState(inst)
					transitiondata = { perfect = true }
				elseif data.control == "skill" then
					nextstate = GetBackfirePreState(inst)
					transitiondata = { perfect = true }
				elseif data.control == "heavyattack" then
					-- Mortar
					nextstate = GetMortarPreState(inst)
					transitiondata = { perfect = true }
				end

				inst.sg.statemem.nextstate = nextstate
				inst.sg.statemem.transitiondata = transitiondata
			end),

			EventHandler("animover", function(inst)
				if inst.sg.statemem.nextstate then -- If they already clicked a button
					inst.sg:GoToState(inst.sg.statemem.nextstate, inst.sg.statemem.transitiondata)
				else
					inst.sg:GoToState("cannon_plant_hold")
				end
			end),
		},
	}),

	State({
		name = "blast_TO_plant",
		tags = { "busy", "airborne" },

		onenter = function(inst, transitiondata)
			-- transitiondata =
			-- {
			-- 	maxspeed = the overall speed of this blast backwards
			-- 	speed = the speed we are currently moving
			-- 	framessliding = how many frames we have already been sliding, so we can continue the same slide
			-- }
			inst.AnimState:PlayAnimation("cannon_H_to_plant")
			inst.sg.statemem.perfectwindow = false
			inst.sg.statemem.earlywindow = true

			-- "Dodge" stuff:
			CheckIfDodging(inst, transitiondata)
			--
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst, true)
		end,

		timeline =
		{
			FrameEvent(6, function(inst)
				-- SGCommon.Fns.BlinkAndFadeColor(inst, {0.25, 0.25, 0.25}, 6) -- 2f eligibility here, 4f eligibility in the hold frame
				inst.sg.statemem.earlywindow = false
				inst.sg.statemem.perfectwindow = true
			end),

			FrameEvent(10, function(inst)
				inst.sg:RemoveStateTag("airborne")
			end),
		},

		onexit = function(inst)
			inst.Physics:Stop()
		end,

		events =
		{
			EventHandler("controlevent", function(inst, data)
				local nextstate
				local transitiondata
				if data.control == "dodge" then
					-- Reload
					nextstate = GetReloadState(inst)
				elseif data.control == "lightattack" then
					nextstate = GetShockwavePreState(inst)
					transitiondata = inst.sg.statemem.perfectwindow
				elseif data.control == "skill" then
					nextstate = GetBackfirePreState(inst)
					transitiondata = { perfect = true } -- This state is trying out a new animation requirement, needs different data
				elseif data.control == "heavyattack" then
					-- Mortar
					nextstate =	GetMortarPreState(inst)
					transitiondata = inst.sg.statemem.perfectwindow
				end

				inst.sg.statemem.nextstate = nextstate
				inst.sg.statemem.transitiondata = transitiondata
			end),

			EventHandler("animover", function(inst)
				if inst.sg.statemem.nextstate then -- If they already clicked a button
					inst.sg:GoToState(inst.sg.statemem.nextstate, inst.sg.statemem.transitiondata)
				else
					inst.sg:GoToState("cannon_plant_hold")
				end
			end),
		},
	}),

	State({
		name = "cannon_plant_hold",
		tags = { "busy" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_atk2_hold_loop", true)
			inst.sg.statemem.dodgecombostate = "planted_reload_fast"
			inst.sg.statemem.lightcombostate = "shockwave_ammocheck"
			inst.sg.statemem.heavycombostate = "mortar_ammocheck"
			inst.sg.statemem.skillcombostate = "backfire_ammocheck"
		end,

		timeline =
		{
			FrameEvent(2, function(inst)
				inst.sg.statemem.dodgecombostate = "planted_reload_slow_late"
				inst.sg.statemem.lightcombostate = "shockwave_pre_late"
				inst.sg.statemem.heavycombostate = "mortar_pre_late"
				if GetRemainingAmmo(inst) > 0 then
					inst.sg.statemem.skillcombostate = "backfire_pre_late"
				else
					inst.sg.statemem.skillcombostate = "backfire_pre_late_noammo" -- The pre_late anim has the player jumping on the cannon, which we don't want to show.
				end
				inst.sg.statemem.canwalkcancel = true
			end),

		},

		onupdate = function(inst)
			if inst.sg.statemem.canwalkcancel and inst.components.playercontroller:GetAnalogDir() ~= nil then
				inst.sg:GoToState("cannon_plant_pst")
			end
		end,

		events =
		{
			EventHandler("animover", function(inst, data)
				inst.sg:GoToState("cannon_plant_pst")
			end),
		},
	}),

	State({
		name = "cannon_plant_pst",
		tags = { "busy" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_atk2_pst", true)
			inst.sg.statemem.dodgecombostate = "planted_reload_slow_late"
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst, data)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "mortar_ammocheck",
		tags = { "busy" },

		onenter = function(inst, perfect)
			local ammo = GetRemainingAmmo(inst)
			local max_ammo = GetMaxAmmo(inst)
			local percent = (ammo > 0 and inst.sg.mem.cannon_override_mortar_ammopercent) or (ammo / max_ammo)

			local mortar_data = { perfect = perfect, ammo_percent = percent }
			if percent > 0 then
				inst.sg:GoToState("mortar_hold", mortar_data)
			else -- No ammo
				inst.sg:GoToState("mortar_noammo")
			end
		end,
	}),

	State({
		name = "mortar_noammo",
		tags = { "busy" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_atk2_pst")
		end,

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "mortar_hold",
		tags = { "busy", "heavy_attack" },

		onenter = function(inst, mortar_data)
			-- mortar_data
			--[[
				perfect: whether or not this was a perfectly timed attack. Focus hit if so!
				ammo_percent: how much ammo they have left, 0-1
			]]
			inst.AnimState:PlayAnimation("cannon_atk2_hold_loop", true)
			inst.sg.statemem.mortar_data = mortar_data or { perfect = false, ammo_percent = 1 }
			CreateMortarAimReticles(inst)
		end,

		timeline =
		{
		},

		onupdate = function(inst)
			if not inst.components.playercontroller:IsControlHeld("heavyattack") then
				inst.sg:GoToState("mortar_shoot", inst.sg.statemem.mortar_data)
			else
				UpdateMortarAimReticles(inst)
			end
		end,

		events =
		{
			EventHandler("controlupevent", function(inst, data)
				if data.control == "heavyattack" then
					inst.sg:GoToState("mortar_shoot", inst.sg.statemem.mortar_data)
				end
			end),
			EventHandler("controlevent", function(inst, data)
				if data.control == "dodge" or data.control == "lightattack" or data.control == "potion" or data.control == "skill" then
					inst.sg:GoToState("cannon_plant_pst")
				end
			end),
		},

		onexit = function(inst)
			inst:DoTaskInAnimFrames(9, DestroyMortarAimReticles) -- Delay some frames so there isn't a gap where no aim reticles exist
		end,
	}),

	State({
		name = "mortar_pre_early",
		tags = { "busy", "heavy_attack" },

		onenter = function(inst, mortar_data) --perfect -- whether or not this mortar shot was started with perfect timing -- aka, becomes a focus hit
			inst.AnimState:PlayAnimation("cannon_mortar_early")
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("mortar_ammocheck", false)
			end)
		},
	}),

	State({
		name = "mortar_pre_late",
		tags = { "busy", "heavy_attack" },

		onenter = function(inst) --perfect -- whether or not this mortar shot was started with perfect timing -- aka, becomes a focus hit
			inst.AnimState:PlayAnimation("cannon_mortar_late")
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("mortar_ammocheck", false)
			end)
		},
	}),

	State({
		name = "mortar_shoot",
		tags = { "busy", "heavy_attack" },

		onenter = function(inst, mortar_data)
			-- mortar_data
			--[[
				perfect: whether or not this was a perfectly timed attack. Focus hit if so!
				ammo_percent: how much ammo they have left, 0-1
			]]

			inst.Transform:SetRotation(inst.Transform:GetFacing() == FACING_LEFT and -180 or 0)

			local perfect = mortar_data and mortar_data.perfect or false
			local ammo = mortar_data and mortar_data.ammo_percent or 1
			local nextstate
			-- 6/6 ammo = strong
			-- 5/6 ammo = strong
			-- 4/6 ammo = medium
			-- 3/6 ammo = medium
			-- 2/6 ammo = weak
			-- 1/6 ammo = weak
			if ammo >= 0.8 then
				nextstate = "mortar_shoot_strong"
			elseif ammo >= 0.5 then
				nextstate = "mortar_shoot_medium"
			else
				nextstate = "mortar_shoot_weak"
			end

			inst.sg:GoToState(nextstate, perfect)
		end,

		timeline =
		{
		},

		events =
		{
		},
	}),

	State({
		name = "mortar_shoot_weak",
		tags = { "busy", "heavy_attack" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")

			inst.AnimState:PlayAnimation("cannon_atk2_shoot")
			inst.AnimState:PushAnimation("cannon_atk2_pst")

			inst.sg.mem.attack_id = "MORTAR_WEAK"
			inst.sg.mem.attack_type = "heavy_attack"

			inst.sg.statemem.weightmult = GetWeightVelocityMult(inst)

			inst.sg.statemem.ammotospend = inst.sg.mem.cannon_override_mortar_ammopershot or GetRemainingAmmo(inst)

			local focus = inst.sg.mem.mortar_focus_sequence[GetRemainingAmmo(inst)]
			DoMortarFX(inst, "fx_player_cannon_mortar_weak_atk", focus)
			ParticleSystemHelper.MakeEventSpawnParticles(inst, {
				duration=60.0,
				offx=1.0,
				offy=1.0,
				offz=0.0,
				particlefxname = focus and "cannon_shot_mortar_focus" or "cannon_shot_mortar",
				use_entity_facing=true,
			})
		end,

		timeline =
		{
			--sound
			FrameEvent(1, function(inst)
				PlayMortarSound(inst, inst.sg.statemem.ammotospend, 0)
			end),
			FrameEvent(3, function(inst)
				DoMortar(inst, inst.sg.statemem.ammotospend)

				inst.sg.statemem.previous_ammo = inst.sg.mem.ammo

				UpdateAmmo(inst, inst.sg.statemem.ammotospend)

				if inst.sg.statemem.ammotospend < 2 then
					local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.mortar_focus_sequence[inst.sg.statemem.previous_ammo])
					PlayMortarTubeSound(inst, 0, isFocusAttack)
					if inst.sg.mem.ammo <= inst.sg.statemem.previous_ammo then
						inst:DoTaskInAnimFrames(4, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
					end
				end
			end),

			FrameEvent(4, function(inst)
				if inst.sg.statemem.ammotospend >= 2 then
					local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.mortar_focus_sequence[inst.sg.statemem.previous_ammo])
					PlayMortarTubeSound(inst, 0, isFocusAttack)
					if inst.sg.mem.ammo <= inst.sg.statemem.previous_ammo then
						inst:DoTaskInAnimFrames(4, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
					end
				end
			end),

			FrameEvent(0, function(inst)
				inst.components.hitstopper:PushHitStop(MORTAR_SHOOT_HITSTOP.WEAK)
			end),
			FrameEvent(1, function(inst)
				DoMortarWeakKickback(inst)
				inst.Physics:SetMotorVel(-MORTAR_SHOOT_BLOWBACK.WEAK * inst.sg.statemem.weightmult)
			end),
			FrameEvent(2, function(inst)
				inst.Physics:SetMotorVel(-MORTAR_SHOOT_BLOWBACK.WEAK * inst.sg.statemem.weightmult * 0.5)
			end),
			FrameEvent(4, function(inst)
				inst.Physics:Stop()
			end),
		},

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),
		},

		onexit = function(inst)
		end,
	}),

	State({
		name = "mortar_shoot_medium",
		tags = { "busy", "heavy_attack" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")

			inst.AnimState:PlayAnimation("cannon_mortar_med_atk")
			inst.sg.mem.attack_id = "MORTAR_MEDIUM"
			inst.sg.mem.attack_type = "heavy_attack"

			inst.sg.statemem.ammotospend = inst.sg.mem.cannon_override_mortar_ammopershot or GetRemainingAmmo(inst)
		end,

		timeline =
		{
			--sound
			FrameEvent(1, function(inst)
				local focus = inst.sg.mem.mortar_focus_sequence[GetRemainingAmmo(inst)]
				DoMortarFX(inst, "fx_player_cannon_mortar_med_atk", focus)
				ParticleSystemHelper.MakeEventSpawnParticles(inst, {
					duration=60.0,
					offx=1.0,
					offy=1.0,
					offz=0.0,
					particlefxname= inst.sg.mem.focus_sequence[inst.sg.statemem.ammotospend] and "cannon_shot_mortar_focus" or "cannon_shot_mortar",
					use_entity_facing=true,
				})
			end),

			FrameEvent(1, function(inst)
				inst.components.hitstopper:PushHitStop(MORTAR_SHOOT_HITSTOP.MEDIUM)
				DoMortar(inst, inst.sg.statemem.ammotospend)

				PlayMortarSound(inst, inst.sg.statemem.ammotospend, 1)

				inst.sg.statemem.previous_ammo = inst.sg.mem.ammo

				UpdateAmmo(inst, inst.sg.statemem.ammotospend)
			end),

			FrameEvent(3, function(inst)
				DoMortarMediumKickback(inst)
			end),

			FrameEvent(4, function(inst)
				local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.mortar_focus_sequence[inst.sg.statemem.previous_ammo])
				PlayMortarTubeSound(inst, 0, isFocusAttack)
				if inst.sg.mem.ammo <= inst.sg.statemem.previous_ammo then
					inst:DoTaskInAnimFrames(4, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
				end
			end),

			FrameEvent(12, function(inst)
				inst.Physics:Stop()
			end),

			--CANCELS
			FrameEvent(23, SGPlayerCommon.Fns.RemoveBusyState),

		},

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),
		},

		onexit = function(inst)
		end,
	}),

	State({
		name = "mortar_shoot_strong",
		tags = { "busy", "heavy_attack" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")
			inst.AnimState:PlayAnimation("cannon_mortar_heavy_atk")
			inst.sg.mem.attack_id = "MORTAR_STRONG"
			inst.sg.mem.attack_type = "heavy_attack"

			inst.sg.statemem.ammotospend = inst.sg.mem.cannon_override_mortar_ammopershot or GetRemainingAmmo(inst)

			inst.sg.statemem.speed = MORTAR_SHOOT_BLOWBACK.STRONG * GetWeightVelocityMult(inst)
		end,

		timeline =
		{
			--sound
			FrameEvent(1, function(inst)
				PlayMortarSound(inst, inst.sg.statemem.ammotospend, 2)
			end),
			FrameEvent(2, function(inst)
				local focus = inst.sg.mem.mortar_focus_sequence[GetRemainingAmmo(inst)]
				DoMortarFX(inst, "fx_player_cannon_mortar_heavy_atk", focus)
				ParticleSystemHelper.MakeEventSpawnParticles(inst, {
					duration=60.0,
					offx=1.0,
					offy=1.0,
					offz=0.0,
					particlefxname = "cannon_shot_mortar",
					use_entity_facing=true,
				})
			end),

			FrameEvent(3, function(inst)
				DoMortar(inst, inst.sg.statemem.ammotospend)
				inst.sg.statemem.previous_ammo = inst.sg.mem.ammo
				UpdateAmmo(inst, inst.sg.statemem.ammotospend)
			end),
			FrameEvent(5, function(inst)
				inst.components.hitstopper:PushHitStop(MORTAR_SHOOT_HITSTOP.STRONG)

				local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.mortar_focus_sequence[inst.sg.statemem.previous_ammo])
				PlayMortarTubeSound(inst, 0, isFocusAttack)

				if inst.sg.mem.ammo <= inst.sg.statemem.previous_ammo then
					inst:DoTaskInAnimFrames(4, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
				end
			end),
			FrameEvent(6, function(inst)
				DoMortarStrongKickback(inst)
			end),
			FrameEvent(9, function(inst)
				inst.Physics:SetMotorVel(-inst.sg.statemem.speed)
			end),
			FrameEvent(12, function(inst)
				inst.Physics:SetMotorVel(-inst.sg.statemem.speed * 0.5)
				inst.sg:AddStateTag("prone")
			end),
			FrameEvent(16, function(inst)
				inst.Physics:Stop()
			end),

			FrameEvent(55, function(inst)
				inst.sg:RemoveStateTag("prone")
			end),

			--CANCELS
			FrameEvent(65, SGPlayerCommon.Fns.SetCanAttackOrAbility),
			FrameEvent(68, SGPlayerCommon.Fns.RemoveBusyState),

		},

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "shockwave_ammocheck",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst, perfect)
			local ammo = GetRemainingAmmo(inst)
			local max_ammo = GetMaxAmmo(inst)
			local percent = ammo / max_ammo

			local shockwave_data = { perfect = perfect, ammo_percent = percent }
			if percent > 0 then
				inst.sg:GoToState("shockwave_hold", shockwave_data)
			else -- No ammo
				inst.sg:GoToState("shockwave_noammo")
			end
		end,
	}),

	State({
		name = "shockwave_noammo",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_atk2_pst")
		end,

		timeline = {
			FrameEvent(6, function(inst)
				PlayNoAmmoSound(inst)
			end),
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "shockwave_hold",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst, shockwave_data)
			-- shockwave_data
			--[[
				perfect: whether or not this was a perfectly timed attack. Focus hit if so!
				ammo_percent: how much ammo they have left, 0-1
			]]
			inst.AnimState:PlayAnimation("cannon_shockwave_hold_loop", false)
			inst.sg.statemem.shockwave_data = shockwave_data or { perfect = false, ammo_percent = 1 }

			soundutil.PlayCodeSound(inst,fmodtable.Event.Cannon_shockwave_hold_LP, {
					max_count = 1,
					is_autostop = true,
					stopatexitstate = true
				})

		end,

		timeline = {},

		onupdate = function(inst)
			if not inst.components.playercontroller:IsControlHeld("lightattack") then
				local params = {}
				params.fmodevent = fmodtable.Event.Cannon_shockwave_plug
				soundutil.PlaySoundData(inst, params)
				inst.sg:GoToState("shockwave_shoot", inst.sg.statemem.shockwave_data)
			end
		end,

		events =
		{
			EventHandler("controlupevent", function(inst, data)
				if data.control == "lightattack" then
				elseif data.control == "dodge" or data.control == "heavyattack" or data.control == "potion" or data.control == "skill" then
					inst.sg:GoToState("cannon_plant_pst")
				end
			end),
		},

		onexit = function(inst)
		end,
	}),

	State({
		name = "shockwave_pre_early",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst, shockwave_data) --perfect -- whether or not this mortar shot was started with perfect timing -- aka, becomes a focus hit
			inst.AnimState:PlayAnimation("cannon_shockwave_early")
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("shockwave_ammocheck", false)
			end)
		},
	}),

	State({
		name = "shockwave_pre_late",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst) --perfect -- whether or not this mortar shot was started with perfect timing -- aka, becomes a focus hit
			inst.AnimState:PlayAnimation("cannon_shockwave_late")
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("shockwave_ammocheck", false)
			end)
		},
	}),

	State({
		name = "shockwave_shoot",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst, shockwave_data)
			-- shockwave_data
			--[[
				perfect: whether or not this was a perfectly timed attack. Focus hit if so!
				ammo_percent: how much ammo they have left, 0-1
			]]

			inst.Transform:SetRotation(inst.Transform:GetFacing() == FACING_LEFT and -180 or 0)

			local perfect = shockwave_data and shockwave_data.perfect or false
			local ammo = shockwave_data and shockwave_data.ammo_percent or 1
			local nextstate
			-- 6/6 ammo = strong
			-- 5/6 ammo = strong
			-- 4/6 ammo = medium
			-- 3/6 ammo = medium
			-- 2/6 ammo = weak
			-- 1/6 ammo = weak
			if ammo >= 0.8 then
				nextstate = "shockwave_shoot_strong"
			elseif ammo >= 0.5 then
				nextstate = "shockwave_shoot_medium"
			else
				nextstate = "shockwave_shoot_weak"
			end

			inst.sg:GoToState(nextstate, perfect)
		end,

		timeline =
		{
		},

		events =
		{
		},
	}),

	State({
		name = "shockwave_shoot_weak",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")

			inst.AnimState:PlayAnimation("cannon_shockwave_shoot")
			inst.AnimState:PushAnimation("cannon_shockwave_pst")

			inst.sg.mem.attack_id = "SHOCKWAVE_WEAK"
			inst.sg.mem.attack_type = "light_attack"

			inst.sg.statemem.shockwave_ammo = GetRemainingAmmo(inst)
		end,

		timeline =
		{
			FrameEvent(1, function(inst)
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname= "fx_cannon_sphere_aoe_focus",
					offz=-0.1,
					scalex=0.8,
					scalez=1.0,
				})
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname= "cannon_aoe_explosion_med_focus",
					offx=0.19,
					offy=0.45,
					offz=-0.1,
				})
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname= "fx_cannon_smoke_aoe_focus",
					offz=-0.1,
					scalex=1.2,
					scalez=1.2,
				})

				ParticleSystemHelper.MakeEventSpawnParticles(inst, {
					duration=90.0,
					particlefxname= "cannon_burst_aoe_sphere_sml_focus",
				})
			end),

			FrameEvent(3, function(inst)
				combatutil.StartMeleeAttack(inst)
				inst.components.hitbox:PushCircle(0, 0, ATTACKS.SHOCKWAVE_WEAK.RADIUS, HitPriority.MOB_DEFAULT)

				DoShockwaveSelfAttack(inst)
				UpdateAmmo(inst, GetRemainingAmmo(inst))
				
				--sound
				local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.focus_sequence[inst.sg.statemem.shockwave_ammo])
				soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_shockwave_shoot_weak, {
					max_count = 1,
					fmodparams = {
						isFocusAttack = isFocusAttack,
						cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
					}
				})

				local previous_ammo = inst.sg.statemem.shockwave_ammo
				if inst.sg.mem.ammo <= previous_ammo then
					inst:DoTaskInAnimFrames(4, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
				end

			end),

			FrameEvent(4, function(inst)
				combatutil.EndMeleeAttack(inst)
			end),

			--CANCELS
			FrameEvent(5, SGPlayerCommon.Fns.SetCanDodge),
			FrameEvent(12, SGPlayerCommon.Fns.SetCanAttackOrAbility),
			FrameEvent(14, SGPlayerCommon.Fns.RemoveBusyState),
		},

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),

			EventHandler("hitboxtriggered", OnShockwaveHitBoxTriggered),
		},

		onexit = function(inst)
		end,
	}),

	State({
		name = "shockwave_shoot_medium",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")

			inst.AnimState:PlayAnimation("cannon_shockwave_med_atk")
			inst.sg.mem.attack_id = "SHOCKWAVE_MEDIUM"
			inst.sg.mem.attack_type = "light_attack"

			inst.sg.statemem.shockwave_ammo = GetRemainingAmmo(inst)
		end,

		timeline =
		{
			FrameEvent(1, function(inst)
				local focus = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)]
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname= focus and "fx_cannon_sphere_aoe_focus" or "fx_cannon_sphere_aoe",
					offz=-0.1,
					scalex=1.0,
					scalez=1.0,
				})
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname=focus and "cannon_aoe_explosion_med_focus" or "cannon_aoe_explosion_med",
					offx=0.19,
					offy=0.45,
					offz=-0.1,
					scalex=1.5,
					scalez=1.5,
				})
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname= focus and "fx_cannon_smoke_aoe_focus" or "fx_cannon_smoke_aoe",
					offz=-0.1,
					scalex=1.5,
					scalez=1.5,
				})

				ParticleSystemHelper.MakeEventSpawnParticles(inst, {
					duration=90.0,
					particlefxname= focus and "cannon_burst_aoe_sphere_med_focus" or "cannon_burst_aoe_sphere_med",
				})
			end),

			FrameEvent(3, function(inst)
				combatutil.StartMeleeAttack(inst)
				inst.components.hitbox:PushCircle(0, 0, ATTACKS.SHOCKWAVE_MEDIUM.RADIUS, HitPriority.MOB_DEFAULT)

				DoShockwaveSelfAttack(inst)
				UpdateAmmo(inst, GetRemainingAmmo(inst))

				--sound
				local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.focus_sequence[inst.sg.statemem.shockwave_ammo])
				soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_shockwave_shoot_medium, {
					max_count = 1,
					fmodparams = {
						isFocusAttack = isFocusAttack,
						cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
					}
				})

				local previous_ammo = inst.sg.statemem.shockwave_ammo
				if inst.sg.mem.ammo <= previous_ammo then
					inst:DoTaskInAnimFrames(4, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
				end

			end),

			FrameEvent(4, function(inst)
				combatutil.EndMeleeAttack(inst)
			end),

			--CANCELS
			FrameEvent(17, SGPlayerCommon.Fns.SetCanDodge),
			FrameEvent(24, SGPlayerCommon.Fns.SetCanAttackOrAbility),
			FrameEvent(26, SGPlayerCommon.Fns.RemoveBusyState),
		},

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),

			EventHandler("hitboxtriggered", OnShockwaveHitBoxTriggered),
		},

		onexit = function(inst)
		end,
	}),

	State({
		name = "shockwave_shoot_strong",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")
			inst.AnimState:PlayAnimation("cannon_shockwave_heavy_atk")
			inst.sg.mem.attack_id = "SHOCKWAVE_STRONG"
			inst.sg.mem.attack_type = "light_attack"

			inst.sg.statemem.shockwave_ammo = GetRemainingAmmo(inst)
		end,

		timeline =
		{
			FrameEvent(1, function(inst)
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname= "fx_cannon_sphere_aoe",
					offz=-0.1,
					scalex=1.5,
					scalez=1.5,
				})
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname= "cannon_mortar_explosion",
					offz=-0.1,
					scalex=1.5,
					scalez=1.5,
				})
				EffectEvents.MakeEventSpawnEffect(inst, {
					fxname="fx_cannon_smoke_aoe",
					offz=-0.1,
					scalex=2.0,
					scalez=2.0,
				})

				ParticleSystemHelper.MakeEventSpawnParticles(inst, {
					particlefxname="cannon_burst_aoe_sphere_lrg",
					duration=90.0,
				})
			end),

			FrameEvent(3, function(inst)
				combatutil.StartMeleeAttack(inst)
				inst.components.hitbox:PushCircle(0, 0, ATTACKS.SHOCKWAVE_STRONG.RADIUS, HitPriority.MOB_DEFAULT)

				UpdateAmmo(inst, GetRemainingAmmo(inst))
				
				--sound
				local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.focus_sequence[inst.sg.statemem.shockwave_ammo])
				soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_shockwave_shoot_strong, {
					max_count = 1,
					fmodparams = {
						isFocusAttack = isFocusAttack,
						cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
					}
				})

				local previous_ammo = inst.sg.statemem.shockwave_ammo
				if inst.sg.mem.ammo <= previous_ammo then
					inst:DoTaskInAnimFrames(7, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
				end
			end),

			FrameEvent(4, function(inst)
				combatutil.EndMeleeAttack(inst)
			end),

			FrameEvent(34, function(inst) DoShockwaveSelfAttack(inst) end),

			--CANCELS
			FrameEvent(65, SGPlayerCommon.Fns.SetCanAttackOrAbility),
			FrameEvent(68, SGPlayerCommon.Fns.RemoveBusyState),
		},

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),

			EventHandler("hitboxtriggered", OnShockwaveHitBoxTriggered),
		},
	}),

	State({
		name = "backfire_ammocheck",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst, timing_data)
			local ammo = GetRemainingAmmo(inst)
			local max_ammo = GetMaxAmmo(inst)
			local percent = ammo / max_ammo

			local perfect = timing_data and timing_data.perfect or true -- If we received no timing data, we know it was perfect because we came right from the "skillcombostate" of cannon_plant_hold
			local backfire_data = { perfect = perfect, ammo_percent = percent }
			if percent > 0 then
				if timing_data and timing_data.poortiming then
					inst.sg:GoToState("backfire_hold_poortiming", backfire_data) -- This state has some different animation requirements.
				else
					inst.sg:GoToState("backfire_hold", backfire_data)
				end
			else -- No ammo
				inst.sg:GoToState("backfire_noammo") -- TODO #backfire need a no ammo?
			end
		end,
	}),

	State({
		name = "backfire_noammo",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_atk2_pst")
		end,
		
		timeline = {
			FrameEvent(2, function(inst)
				PlayNoAmmoSound(inst)
			end),
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "backfire_hold",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst, backfire_data)
			-- backfire_data
			--[[
				perfect: whether or not this was a perfectly timed attack. Focus hit if so!
				ammo_percent: how much ammo they have left, 0-1
			]]
			inst.AnimState:PlayAnimation("cannon_backfire_hold_pre", false)
			inst.AnimState:PushAnimation("cannon_backfire_hold_loop", false)

			soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_backfire_hold_LP, {
				max_count = 1,
				is_autostop = true,
				stopatexitstate = true
			})

			inst.sg.statemem.backfire_data = backfire_data or { perfect = false, ammo_percent = 1 }
		end,

		timeline =
		{
			-- TODO #backfire this should be about 3
			FrameEvent(6, function(inst) -- TODO #backfire adjust this once anim is smoothed out -- should let the first frame of the second pose through, then transition
				inst.sg.statemem.can_shoot = true -- If we release the button after this point, we can transition.

				if inst.sg.statemem.shoot_requested then
					-- However, if we've already released the button before this point -- go directly into the shoot.
					inst.sg:GoToState("backfire_shoot", inst.sg.statemem.backfire_data)
				end
			end),
		},

		onupdate = function(inst)
			if not inst.components.playercontroller:IsControlHeld("skill") then
				if inst.sg.statemem.can_shoot then
					inst.sg:GoToState("backfire_shoot", inst.sg.statemem.backfire_data)
				else
					inst.sg.statemem.shoot_requested = true
				end
			end
		end,

		events =
		{
			EventHandler("controlupevent", function(inst, data)
				if data.control == "skill" then
					if inst.sg.statemem.can_shoot then
						-- Don't allow going directly to this state -- force us to wait at least to the "can_shoot" flag is set.
						inst.sg:GoToState("backfire_shoot", inst.sg.statemem.backfire_data)
					else
						inst.sg.statemem.shoot_requested = true
					end
				elseif data.control == "dodge" or data.control == "heavyattack" or data.control == "potion" or data.control == "lightattack" then
					inst.sg:GoToState("cannon_plant_pst") -- TODO #backfire need a cancel out
				end
			end),
		},
	}),

	State({
		name = "backfire_hold_poortiming",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst, backfire_data)
			-- backfire_data
			--[[
				perfect: whether or not this was a perfectly timed attack. Focus hit if so!
				ammo_percent: how much ammo they have left, 0-1
			]]
			inst.AnimState:PlayAnimation("cannon_backfire_hold_loop", false)
			inst.sg.statemem.backfire_data = backfire_data or { perfect = false, ammo_percent = 1 }


			soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_backfire_hold_LP, {
				max_count = 1,
				is_autostop = true,
				stopatexitstate = true
			})

		end,

		timeline =
		{
			FrameEvent(4, function(inst)
				inst.sg.statemem.can_shoot = true -- If we release the button after this point, we can transition.

				if inst.sg.statemem.shoot_requested then
					-- However, if we've already released the button before this point -- go directly into the shoot.
					inst.sg:GoToState("backfire_shoot", inst.sg.statemem.backfire_data)
				end
			end),
		},

		onupdate = function(inst)
			if not inst.components.playercontroller:IsControlHeld("skill") then
				if inst.sg.statemem.can_shoot then
					inst.sg:GoToState("backfire_shoot", inst.sg.statemem.backfire_data)
				else
					inst.sg.statemem.shoot_requested = true
				end
			end
		end,

		events =
		{
			EventHandler("controlupevent", function(inst, data)
				if data.control == "skill" then
					if inst.sg.statemem.can_shoot then
						-- Don't allow going directly to this state -- force us to wait at least to the "can_shoot" flag is set.
						inst.sg:GoToState("backfire_shoot", inst.sg.statemem.backfire_data)
					else
						inst.sg.statemem.shoot_requested = true
					end
				elseif data.control == "dodge" or data.control == "heavyattack" or data.control == "potion" or data.control == "lightattack" then
					inst.sg:GoToState("cannon_plant_pst") -- TODO #backfire need a cancel out
				end
			end),
		},
	}),

	State({
		name = "backfire_pre_early",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_backfire_early")
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("backfire_ammocheck", { perfect = false, poortiming = true })
			end)
		},
	}),

	State({
		name = "backfire_pre_early_noammo",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_shockwave_early") -- Reuse anim, which doesn't have us hopping on the cannon.
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("backfire_ammocheck", { perfect = false, poortiming = true })
			end)
		},
	}),


	State({
		name = "backfire_pre_late",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_backfire_late")
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("backfire_ammocheck", { perfect = false, poortiming = true })
			end)
		},
	}),

	State({
		name = "backfire_pre_late_noammo",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("cannon_shockwave_late") -- Reuse anim, which doesn't have us hopping on the cannon.
		end,

		timeline =
		{
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("backfire_ammocheck", { perfect = false, poortiming = true })
			end)
		},
	}),

	State({
		name = "backfire_shoot",
		tags = { "busy", "nointerrupt", "light_attack" },

		onenter = function(inst, backfire_data)
			-- backfire_data
			--[[
				perfect: whether or not this was a perfectly timed attack. Focus hit if so!
				ammo_percent: how much ammo they have left, 0-1
			]]

			inst.Transform:SetRotation(inst.Transform:GetFacing() == FACING_LEFT and -180 or 0)

			local perfect = backfire_data and backfire_data.perfect or false
			local ammo = backfire_data and backfire_data.ammo_percent or 1
			local nextstate
			-- 6/6 ammo = strong
			-- 5/6 ammo = strong
			-- 4/6 ammo = medium
			-- 3/6 ammo = medium
			-- 2/6 ammo = weak
			-- 1/6 ammo = weak
			if ammo >= 0.8 then
				nextstate = "backfire_shoot_strong"
			elseif ammo >= 0.5 then
				nextstate = "backfire_shoot_medium"
			else
				nextstate = "backfire_shoot_weak"
			end

			inst.sg:GoToState(nextstate, perfect)
		end,

		timeline =
		{
		},

		events =
		{
		},
	}),

	State({
		name = "backfire_shoot_weak",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")

			inst.AnimState:PlayAnimation("cannon_backfire_shoot")
			inst.AnimState:PushAnimation("cannon_backfire_pst")

			inst.sg.mem.attack_id = "BACKFIRE_WEAK"
			inst.sg.mem.attack_type = "skill"
			inst.sg.statemem.backfire_ammo = GetRemainingAmmo(inst)

			DoShotFX(inst, nil, "fx_player_cannon_backfire_light") -- Always play this FX specifically for this backfire

			inst.sg.statemem.speed = BACKFIRE_VELOCITY.WEAK * GetBackfireWeightVelocityMult(inst)

			--fx
			ParticleSystemHelper.MakeEventSpawnParticles(inst, {
				duration=20.0,
				ischild=true,
				offx=1.45,
				offy=0.0,
				offz=0.0,
				particlefxname="cannon_backfire_bounce_sparks_focus",
				use_entity_facing=true,
			})
		end,

		timeline =
		{
			--PHYSICS
			-- FrameEvent(0, function(inst) inst.components.hitstopper:PushHitStop(HitStopLevel.LIGHT) end),
			FrameEvent(1, function(inst) DoBackfireWeakKickback(inst) end),
			FrameEvent(1, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 1) end),
			FrameEvent(1, function(inst) inst.Physics:StartPassingThroughObjects() end),

			FrameEvent(2, function(inst) DoShotFX(inst, nil, "fx_player_cannon_backfire_light") end), -- Play another FX now that we're moving -- KA-CLUNK!

			FrameEvent(3, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 1.25) end),

			FrameEvent(6, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.75) end),
			FrameEvent(8, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.5) end),
			FrameEvent(10, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.25) end),
			FrameEvent(12, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.125) end),
			FrameEvent(12, function(inst) SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst) end),
			FrameEvent(14, function(inst)
				inst.Physics:Stop()
			end),

			-- ATTACK DATA
			FrameEvent(1, function(inst)
				combatutil.StartMeleeAttack(inst)
				inst.sg.statemem.hitflags = Attack.HitFlags.LOW_ATTACK
				inst.components.hitbox:StartRepeatTargetDelay()
				inst.components.hitbox:PushBeam(4, 2, 2.5, HitPriority.PLAYER_DEFAULT) -- Thicker in front
				inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT)  -- Thinner in back

				UpdateAmmo(inst, GetRemainingAmmo(inst))

				--sound
				local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.focus_sequence[inst.sg.statemem.backfire_ammo])
				inst.sg.mem.backfire_sound = soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_backfire_shoot_weak,
					{
						max_count = 1,
						fmodparams = {
							isFocusAttack = isFocusAttack,
							cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
						}
					})

				local previous_ammo = inst.sg.statemem.backfire_ammo
				if inst.sg.mem.ammo <= previous_ammo then
					inst:DoTaskInAnimFrames(4, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
				end

			end),
			FrameEvent(2, function(inst) inst.components.hitbox:PushBeam(2, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(3, function(inst) inst.components.hitbox:PushBeam(2, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(4, function(inst) inst.components.hitbox:PushBeam(2, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(5, function(inst) inst.components.hitbox:PushBeam(2, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),

			FrameEvent(6, function(inst) inst.components.hitbox:PushBeam(1.5, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(7, function(inst) inst.components.hitbox:PushBeam(1.5, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(8, function(inst) inst.components.hitbox:PushBeam(1.5, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(9, function(inst) inst.components.hitbox:PushBeam(1.5, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(10, function(inst) inst.components.hitbox:PushBeam(1.5, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(10, function(inst) inst.components.hitbox:PushBeam(1.5, -1.75, 1, HitPriority.PLAYER_DEFAULT) end),

			FrameEvent(10, function(inst)
				inst.components.hitbox:StopRepeatTargetDelay()
				combatutil.EndMeleeAttack(inst)
			end),

			-- --CANCELS
			FrameEvent(15, SGPlayerCommon.Fns.SetCanDodge),
			FrameEvent(18, SGPlayerCommon.Fns.RemoveBusyState),
		},

		onexit = function(inst)
			SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst)
			inst.components.hitbox:StopRepeatTargetDelay()
			if inst.sg.mem.backfire_sound then
				soundutil.KillSound(inst, inst.sg.mem.backfire_sound)
				inst.sg.mem.backfire_sound = nil
			end
		end,

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),

			EventHandler("hitboxtriggered", OnBackfireHitBoxTriggered),
		},
	}),

	State({
		name = "backfire_shoot_medium",
		tags = { "busy", "light_attack", "nointerrupt" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")

			inst.AnimState:PlayAnimation("cannon_backfire_med_atk")
			inst.sg.mem.attack_id = "BACKFIRE_MEDIUM_EARLY"
			inst.sg.mem.attack_type = "skill"
			inst.sg.statemem.backfire_ammo = GetRemainingAmmo(inst)

			local focus = inst.sg.mem.focus_sequence[inst.sg.statemem.backfire_ammo]

			DoShotFX(inst, nil, focus and "fx_player_cannon_backfire_med_focus" or "fx_player_cannon_backfire_med")

			inst.sg.statemem.speed = BACKFIRE_VELOCITY.MEDIUM * GetBackfireWeightVelocityMult(inst)

			--fx
			local param =
			{
				duration = 20.0,
				ischild = true,
				offx = 1.45,
				offy = 0.0,
				offz = 0.0,
				particlefxname = focus and "cannon_backfire_bounce_sparks_focus" or "cannon_backfire_bounce_sparks",
				use_entity_facing=true,
			}
			ParticleSystemHelper.MakeEventSpawnParticles(inst, param)
		end,

		timeline =
		{
			--PHYSICS
			FrameEvent(1, function(inst) inst.components.hitstopper:PushHitStop(HitStopLevel.LIGHT) end),
			FrameEvent(2, function(inst) DoBackfireMediumKickback(inst) end),
			FrameEvent(2, function(inst)
				-- Play the FX again now that we're moving. Ka-KLUNK!
				local focus = inst.sg.mem.focus_sequence[inst.sg.statemem.backfire_ammo]
				DoShotFX(inst, nil, focus and "fx_player_cannon_backfire_med_focus" or "fx_player_cannon_backfire_med")

				inst.Physics:StartPassingThroughObjects()
				inst.Physics:SetMotorVel(inst.sg.statemem.speed * 1)
			end),

			FrameEvent(4, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 1.15) end),

			FrameEvent(9, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.5) end),
			FrameEvent(13, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.25) end),
			FrameEvent(18, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.125) end),
			FrameEvent(18, function(inst) SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst) end),
			FrameEvent(23, function(inst)
				inst.Physics:Stop()
			end),

			-- ATTACK DATA
			FrameEvent(2, function(inst)
				combatutil.StartMeleeAttack(inst)
				inst.sg.statemem.hitflags = Attack.HitFlags.LOW_ATTACK
				inst.components.hitbox:StartRepeatTargetDelay()
				inst.components.hitbox:PushBeam(4, 2, 2.5, HitPriority.PLAYER_DEFAULT) -- Thicker in front
				inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT)  -- Thinner in back

				UpdateAmmo(inst, GetRemainingAmmo(inst))

				--sound
				local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.focus_sequence[inst.sg.statemem.backfire_ammo])
				inst.sg.mem.backfire_sound = soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_backfire_shoot_medium,
					{
						max_count = 1,
						fmodparams = {
							isFocusAttack = isFocusAttack,
							cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
						}
					})

				local previous_ammo = inst.sg.statemem.backfire_ammo
				if inst.sg.mem.ammo <= previous_ammo then
					inst:DoTaskInAnimFrames(4, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
				end

			end),
			FrameEvent(3, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(4, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(5, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(6, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(7, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(8, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(9, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(10, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(11, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(12, function(inst) inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT) end),

			FrameEvent(13, function(inst) inst.sg.mem.attack_id = "BACKFIRE_MEDIUM_LATE" end),
			FrameEvent(13, function(inst) inst.components.hitbox:PushBeam(2, -2, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(14, function(inst) inst.components.hitbox:PushBeam(2, -2, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(15, function(inst) inst.components.hitbox:PushBeam(2, -2, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(16, function(inst) inst.components.hitbox:PushBeam(2, -2, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(17, function(inst) inst.components.hitbox:PushBeam(2, -2, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(18, function(inst) inst.components.hitbox:PushBeam(2, -2, 1, HitPriority.PLAYER_DEFAULT) end),
			FrameEvent(18, function(inst)
				combatutil.EndMeleeAttack(inst)
			end),

			-- --CANCELS
			FrameEvent(17, function(inst)
				inst.components.playercontroller:AddGlobalControlQueueTicksModifier(7, "backfire_shoot_medium")
			end),
			FrameEvent(22, SGPlayerCommon.Fns.SetCanDodge),
			FrameEvent(24, SGPlayerCommon.Fns.SetCanAttackOrAbility),
			FrameEvent(28, SGPlayerCommon.Fns.RemoveBusyState),
		},

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),

			EventHandler("hitboxtriggered", OnBackfireHitBoxTriggered),
		},

		onexit = function(inst)
			inst.components.playercontroller:RemoveGlobalControlQueueTicksModifier("backfire_shoot_medium")
			inst.components.hitbox:StopRepeatTargetDelay()
			SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst)

			if inst.sg.mem.backfire_sound then
				soundutil.KillSound(inst, inst.sg.mem.backfire_sound)
				inst.sg.mem.backfire_sound = nil
			end
		end,
	}),

	State({
		name = "backfire_shoot_strong",
		tags = { "busy", "light_attack", "nointerrupt" },

		onenter = function(inst)
			inst:PushEvent("attack_state_start")
			inst.AnimState:PlayAnimation("cannon_backfire_heavy_atk")
			inst.sg.mem.attack_id = "BACKFIRE_STRONG_EARLY"
			inst.sg.mem.attack_type = "skill"
			inst.sg.statemem.backfire_ammo = GetRemainingAmmo(inst)

			inst.sg.statemem.speed = BACKFIRE_VELOCITY.STRONG * GetBackfireWeightVelocityMult(inst)
			inst.Physics:StartPassingThroughObjects()

			DoShotFX(inst, nil, "fx_player_cannon_backfire_heavy")

			--fx
			local param =
			{
				duration=20.0,
				ischild=true,
				offx=1.45,
				offy=0.0,
				offz=0.0,
				particlefxname="cannon_backfire_bounce_sparks",
				use_entity_facing=true,
			}
			ParticleSystemHelper.MakeEventSpawnParticles(inst, param)
		end,

		timeline =
		{
			--PHYSICS
			FrameEvent(1, function(inst) inst.components.hitstopper:PushHitStop(HitStopLevel.MEDIUM) end),

			FrameEvent(2, function(inst) DoBackfireStrongKickback(inst) end),
			FrameEvent(2, function(inst)
				inst.Physics:StartPassingThroughObjects()
				inst.Physics:SetMotorVel(inst.sg.statemem.speed)
			end),

			FrameEvent(10, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.5) end),
			FrameEvent(18, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.125) end), -- Hit ground
			-- FrameEvent(18, function(inst) inst.components.hitstopper:PushHitStop(HitStopLevel.LIGHT) end), -- Hit ground
			FrameEvent(20, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.25) end), -- Bounce
			FrameEvent(23, function(inst) inst.Physics:SetMotorVel(inst.sg.statemem.speed * 0.125) end),
			FrameEvent(27, function(inst) inst.Physics:Stop() end),
			FrameEvent(27, function(inst) SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst) end),

			-- HITBOX STUFF
			FrameEvent(2, function(inst) inst.sg:AddStateTag("airborne") end),
			FrameEvent(3, function(inst) inst.sg:AddStateTag("airborne_high") end),
			FrameEvent(4, function(inst) inst.HitBox:SetEnabled(false) end),
			FrameEvent(13, function(inst) inst.HitBox:SetEnabled(true) end),
			FrameEvent(15, function(inst) inst.sg:RemoveStateTag("airborne_high") end),
			FrameEvent(18, function(inst) inst.sg:RemoveStateTag("airborne") end),

			FrameEvent(2, function(inst)
				combatutil.StartMeleeAttack(inst)
				inst.components.hitbox:StartRepeatTargetDelay()
				inst.sg.statemem.hitflags = Attack.HitFlags.LOW_ATTACK -- Don't hit airborne high yet.
				inst.components.hitbox:PushBeam(4, 2, 2.5, HitPriority.PLAYER_DEFAULT) -- Thicker in front
				inst.components.hitbox:PushBeam(2, -3, 1, HitPriority.PLAYER_DEFAULT)  -- Thinner in back
				DoBackfireSelfAttack(inst)

				UpdateAmmo(inst, GetRemainingAmmo(inst))

				--sound
				local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.focus_sequence[inst.sg.statemem.backfire_ammo])
				inst.sg.mem.backfire_sound = soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_backfire_shoot_strong,
					{
						max_count = 1,
						fmodparams = {
							isFocusAttack = isFocusAttack,
							cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
						}
					})

				local previous_ammo = inst.sg.statemem.backfire_ammo
				if inst.sg.mem.ammo <= previous_ammo then
					inst:DoTaskInAnimFrames(7, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
				end

				inst.sg.statemem.hitfx_y_offset = 2
			end),

			FrameEvent(3, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitflags = Attack.HitFlags.AIR_HIGH
				inst.sg.statemem.hitfx_y_offset = 3.25
			end),

			FrameEvent(4, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.5
			end),
			FrameEvent(5, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.5
			end),
			FrameEvent(6, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.75
			end),
			FrameEvent(7, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.75
			end),
			FrameEvent(8, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 4
			end),
			FrameEvent(9, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 4.25
			end),
			FrameEvent(10, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 4
			end),
			FrameEvent(11, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.75
			end),
			FrameEvent(12, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.75
			end),
			FrameEvent(13, function(inst)
				inst.components.hitbox:PushBeam(-1, 1, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.5
			end),


			FrameEvent(14, function(inst)
				inst.sg.mem.attack_id = "BACKFIRE_STRONG_LATE"
				inst.sg.statemem.hitflags = Attack.HitFlags.AIR_HIGH
				inst.components.hitbox:PushBeam(0, -2, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.5
			end),
			FrameEvent(15, function(inst)
				inst.sg.statemem.hitflags = nil
				inst.components.hitbox:PushBeam(0, -2, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.5
			end),

			FrameEvent(16, function(inst)
				inst.components.hitbox:PushBeam(0, -2, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3.5
			end),

			FrameEvent(17, function(inst)
				inst.components.hitbox:PushBeam(0, -2, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3
			end),

			FrameEvent(18, function(inst)
				inst.components.hitbox:PushBeam(0, -2, 1, HitPriority.PLAYER_DEFAULT)
				inst.sg.statemem.hitfx_y_offset = 3
				DoBackfireSelfAttack(inst)
				combatutil.EndMeleeAttack(inst)
			end),
			FrameEvent(19, function(inst) inst.components.hitbox:StopRepeatTargetDelay() end),

			-- --CANCELS
			FrameEvent(67, SGPlayerCommon.Fns.RemoveBusyState),
		},

		onexit = function(inst)
			inst.components.hitbox:StopRepeatTargetDelay()
			if inst.sg.mem.backfire_sound then
				soundutil.KillSound(inst, inst.sg.mem.backfire_sound)
				inst.sg.mem.backfire_sound = nil
			end
		end,

		events =
		{
			EventHandler("animqueueover", function(inst, data)
				inst.sg:GoToState("idle")
			end),

			EventHandler("hitboxtriggered", OnBackfireHitBoxTriggered),
		},
	}),

	State({
		name = "planted_reload_fast",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst, attacktype)
			-- SGCommon.Fns.FlickerColor(inst, {.25, .25, .25}, 3, false, true)
			inst.AnimState:PlayAnimation("cannon_reload_fast")
			inst.sg.statemem.attacktype = attacktype

			inst:PushEvent("start_cannonreload_fast")
		end,

		timeline =
		{
			FrameEvent(13, function(inst)
				inst.components.progresstracker:IncrementValue("total_cannon_reloads")
				OnReload(inst, GetMissingAmmo(inst))

				inst.sg.statemem.lightcombostate = "default_light_attack"
				inst.sg.statemem.heavycombostate = "default_heavy_attack"

				SGPlayerCommon.Fns.SetCanDodge(inst)
			end),
		},

		onexit = function(inst)
		end,

		events =
		{
			EventHandler("animover", function(inst, data)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "planted_reload_slow_late",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst, attacktype)
			inst.AnimState:PlayAnimation("cannon_reload_slow")
			inst:PushEvent("start_cannonreload_slow")
		end,

		timeline =
		{

			FrameEvent(19, function(inst) --TODO: retime with actual anim
				inst.components.progresstracker:IncrementValue("total_cannon_reloads")
				OnReload(inst, 3)
			end),

			FrameEvent(31, function(inst) --TODO: retime with actual anim
				OnReload(inst, 3)
			end),

			-- FrameEvent(31, function(inst) --TODO: retime with actual anim
			-- 	inst.sg.statemem.lightcombostate = "default_light_attack"
			-- 	inst.sg.statemem.heavycombostate = "default_heavy_attack"
			-- end),
		},

		events =
		{
			EventHandler("animover", function(inst, data)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "planted_reload_slow_early",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst, attacktype)
			inst.AnimState:PlayAnimation("cannon_reload_early")
			inst:PushEvent("start_cannonreload_early")
		end,

		timeline =
		{

			FrameEvent(32, function(inst) --TODO: retime with actual anim
				inst.components.progresstracker:IncrementValue("total_cannon_reloads")
				OnReload(inst, GetMissingAmmo(inst))
			end),

			-- FrameEvent(31, function(inst) --TODO: retime with actual anim
			-- 	inst.sg.statemem.lightcombostate = "default_light_attack"
			-- 	inst.sg.statemem.heavycombostate = "default_heavy_attack"
			-- end),
		},

		events =
		{
			EventHandler("animover", function(inst, data)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "planted_reload_big_pst",
		tags = { "busy" },

		onenter = function(inst, wasfast)
			inst.AnimState:PlayAnimation("cannon_reload_big_pst")
			inst.sg.statemem.wasfast = wasfast or false
		end,

		timeline =
		{
			FrameEvent(0, function(inst)
				if inst.sg.statemem.wasfast then
					SGPlayerCommon.Fns.RemoveBusyState(inst)
				end
			end)
		},

		events =
		{
			EventHandler("animover", function(inst, data)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "cannon_quickrise",
		tags = { "busy", "heavy_attack", "dodge", "dodging_backwards" },

		onenter = function(inst, real_quickrise)

			inst.AnimState:PlayAnimation("cannon_getup_dodge")
			SGPlayerCommon.Fns.UnsetCanDodge(inst)

			SGPlayerCommon.Fns.SetRollPhysicsSize(inst)
			inst.HitBox:SetInvincible(true)
			SGCommon.Fns.StartJumpingOverHoles(inst)

			if real_quickrise then
				-- We can get here through the normal cannon Heavy combo, so only push the event + present effects when this is a true quickrise.
				inst:PushEvent("quick_rise")
				SGPlayerCommon.Fns.DoCannonQuickRise(inst)
			end

			inst:PushEvent("dodge")
			inst:PushEvent("attack_state_start")
			inst.sg.mem.attack_id = "QUICK_RISE"
			inst.sg.mem.attack_type = "heavy_attack"
			inst.sg.statemem.quickrise_ammo = GetRemainingAmmo(inst)

			-- TODO: commonize this
			local hitstop = TUNING.HITSTOP_PLAYER_QUICK_RISE_FRAMES
			inst.components.hitstopper:PushHitStop(hitstop)
			inst:DoTaskInAnimFrames(hitstop, function()
				if inst ~= nil and inst:IsValid() then
					if GetRemainingAmmo(inst) > 0 then
						inst.Physics:StartPassingThroughObjects()

						combatutil.StartMeleeAttack(inst)

						inst.components.hitbox:PushCircle(0, 0, ATTACKS.QUICKRISE.RADIUS, HitPriority.MOB_DEFAULT)

						local focus = inst.sg.mem.focus_sequence[GetRemainingAmmo(inst)]
						-- fx, lots that would usually be done in embellisher but since focusness matters we'll do it here:
						EffectEvents.MakeEventSpawnEffect(inst, {
							fxname= focus and "cannon_aoe_explosion_med_focus" or "cannon_aoe_explosion_med",
							offx=0.2,
							offy=0.45,
							offz=-0.1,
							scalex=1.0,
							scalez=1.0,	
						})
						EffectEvents.MakeEventSpawnEffect(inst, {
							fxname=focus and "fx_cannon_smoke_aoe_focus" or "fx_cannon_smoke_aoe",
							offx=0.0,
							offy=0.0,
							offz=-0.1,
							scalex=1.2,
							scalez=1.2,
						})

						ParticleSystemHelper.MakeEventSpawnParticles(inst, {
							duration=90.0,
							particlefxname= focus and "cannon_burst_aoe_sphere_sml_focus" or "cannon_burst_aoe_sphere_sml",
						})
						ParticleSystemHelper.MakeEventSpawnParticles(inst, {
							duration=90.0,
							offx=1.2,
							offy=0.0,
							offz=0.0,
							particlefxname= focus and "cannon_shot_quickrise_focus" or "cannon_shot_quickrise",
							render_in_front=true,
							use_entity_facing=true,
						})

						UpdateAmmo(inst, 1)

						-- sound
						local isFocusAttack = soundutil.CoerceFmodParamToNumber(inst.sg.mem.focus_sequence[inst.sg.statemem.quickrise_ammo])
						soundutil.PlayCodeSound(inst, fmodtable.Event.Cannon_shoot_quickrise, {
							max_count = 1,
							fmodparams = {
								isFocusAttack = isFocusAttack,
								cannon_heavyShotType = 2,
								cannon_remainingAmmo_scaled = GetRemainingAmmo(inst) / GetMaxAmmo(inst),
								cannon_ammo_percentDelta = inst.sg.statemem.percent_ammo_changed_param,
							}
						})

						local previous_ammo = inst.sg.statemem.quickrise_ammo
						if inst.sg.mem.ammo and previous_ammo and (inst.sg.mem.ammo <= previous_ammo) then
							inst:DoTaskInAnimFrames(2, function(inst) PlayLowAmmoSound(inst, inst.sg.mem.ammo) end)
						end

						ConfigureNewDodge(inst)
						StartNewDodge(inst)
					end
				end
			end)
		end,

		onupdate = function(inst)
			DoDodgeMovement(inst, true)
		end,

		timeline =
		{
			FrameEvent(1, function(inst)
				combatutil.EndMeleeAttack(inst)
			end),
			FrameEvent(3, function(inst)
				DoQuickRiseKickback(inst)
			end),

			--CANCELS
			FrameEvent(8, function(inst)
				inst.sg.statemem.airplant = true
				if inst.sg.statemem.triedplantearly then
					local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
					inst.sg:GoToState("blast_TO_plant", transitiondata)
				end
			end),
			FrameEvent(11, function(inst)
				inst.sg.statemem.airplant = false
				inst.sg.statemem.groundplant = true
				SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst)
			end),
			FrameEvent(15, SGPlayerCommon.Fns.SetCanAttackOrAbility),
			FrameEvent(20, SGPlayerCommon.Fns.RemoveBusyState),
		},

		onexit = function(inst)
			inst.HitBox:SetInvincible(false)
			SGPlayerCommon.Fns.UndoRollPhysicsSize(inst)
			SGPlayerCommon.Fns.SafeStopPassingThroughObjects(inst)
			inst.Physics:Stop()
			SGCommon.Fns.StopJumpingOverHoles(inst)
		end,


		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),

			EventHandler("controlevent", function(inst, data)
				if data.control == "dodge" then
					local transitiondata = { maxspeed = inst.sg.statemem.maxspeed, speed = inst.sg.statemem.speed, framessliding = inst.sg.statemem.framessliding }
					if inst.sg.statemem.airplant then
						inst.sg:GoToState("blast_TO_plant", transitiondata)
					elseif inst.sg.statemem.groundplant then
						inst.sg:GoToState("cannon_plant_pre", transitiondata)
					else
						inst.sg.statemem.triedplantearly = true
					end
				end
			end),

			EventHandler("hitboxtriggered", OnQuickriseHitBoxTriggered),
		},
	}),

	State({
		name = "cannon_quickrise_noammo",
		tags = { "busy", "heavy_attack" },

		onenter = function(inst, real_quickrise)
			inst.AnimState:PlayAnimation("cannon_getup_dodge")

			if real_quickrise then
				-- We can get here through the normal cannon Heavy combo, so only push the event + present effects when this is a true quickrise.
				inst:PushEvent("quick_rise")
				SGPlayerCommon.Fns.DoCannonQuickRise(inst)
			end
		end,

		onupdate = function(inst)
		end,

		timeline =
		{
			FrameEvent(1, function(inst)
				PlayNoAmmoSound(inst)
			end),
		},

		onexit = function(inst)
		end,


		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},
	}),
}

SGPlayerCommon.States.AddAllBasicStates(states)

return StateGraph("sg_player_cannon", states, events, "init")
